# Copyright 2018 - Nokia
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
import abc

from vitrage.datasources.transformer_base import extract_field_value


class EventFieldExtractor(object):
    """Extract field values from a Nova instance event"""
    @abc.abstractmethod
    def name(self, event):
        pass

    @abc.abstractmethod
    def entity_id(self, event):
        pass

    @abc.abstractmethod
    def state(self, event):
        pass

    @abc.abstractmethod
    def host(self, event):
        pass

    @abc.abstractmethod
    def tenant_id(self, event):
        pass

    @abc.abstractmethod
    def instance_name(self, event):
        pass


class SnapshotEventFieldExtractor(EventFieldExtractor):
    """Extract field values from an event generated by nova.list API call"""
    def name(self, event):
        return extract_field_value(event, 'name')

    def entity_id(self, event):
        return extract_field_value(event, 'id')

    def state(self, event):
        return extract_field_value(event, 'status')

    def host(self, event):
        return extract_field_value(event, 'OS-EXT-SRV-ATTR:host')

    def tenant_id(self, event):
        return extract_field_value(event, 'tenant_id')

    def instance_name(self, event):
        return extract_field_value(event, 'OS-EXT-SRV-ATTR:instance_name')


class LegacyNotificationFieldExtractor(EventFieldExtractor):
    """Extract field values from a Nova legacy notification"""
    def name(self, event):
        return extract_field_value(event, 'hostname')

    def entity_id(self, event):
        return extract_field_value(event, 'instance_id')

    def state(self, event):
        return extract_field_value(event, 'state')

    def host(self, event):
        return extract_field_value(event, 'host')

    def tenant_id(self, event):
        return extract_field_value(event, 'tenant_id')

    def instance_name(self, event):
        return None


class VersionedNotificationFieldExtractor(EventFieldExtractor):
    """Extract field values from a Nova versioned notification"""
    def name(self, event):
        return extract_field_value(event, 'nova_object.data', 'host_name')

    def entity_id(self, event):
        return extract_field_value(event, 'nova_object.data', 'uuid')

    def state(self, event):
        return extract_field_value(event, 'nova_object.data', 'state')

    def host(self, event):
        return extract_field_value(event, 'nova_object.data', 'host')

    def tenant_id(self, event):
        return extract_field_value(event, 'nova_object.data', 'tenant_id')

    def instance_name(self, event):
        return extract_field_value(event, 'nova_object.data', 'instance_name')
