---
title: Asserts
section: 5.01
redirect_from:
  - /asserts/
  - /asserts
---

# Asserts

The `Test` object has a collection of assertion methods.

All assertion methods take optional `message` and `extra` arguments as
the last two params.  The `message` is the name of the test.  The
`extra` argument can contain any arbitrary data about the test, but
the following fields are "special".

* `todo` Set to boolean `true` or a String to mark this as pending
* `skip` Set to boolean `true` or a String to mark this as skipped
* `diagnostic` Set to boolean `true` to show a yaml diagnostic block
  even if the test point passes.  (Failing asserts always show yaml
  diagnostics.)
* `at` Generated by the framework.  The location where the assertion
  was called.  Do not set this field unless you know what you are
  doing.
* `stack` Generated by the framework.  The stack trace to the point
  where the assertion was called.  Do not set this field unless you
  know what you are doing.

**Note**: There's no requirement that you use tap's built-in
assertions.  You can also use any Error-throwing assertion library,
including Node.js's built-in `assert` module.  A throw fails a test,
so not-throwing passes.  That does, however, mean that you won't
generate a test point in your output for each assertion run.  You do
you.

## t.ok(obj, message, extra)

Verifies that the object is truthy.

## t.notOk(obj, message, extra)

Verifies that the object is not truthy.

## t.error(obj, message, extra)

If the object is an error, then the assertion fails.

Note: if an error is encountered unexpectedly, it's often better to
simply throw it.  The Test object will handle this as a failure.

## t.emits(eventEmitter, event, message, extra)

Verify that the event emitter emits the named event before the end of
the test.

Returns a promise that resolves when the event is emitted.

## t.rejects(promise | fn, [expectedError], message, extra)

Verifies that the promise (or promise-returning function) rejects.  If
an expected error is provided, then also verify that the rejection
matches the expected error.

Note: since promises always reject and resolve asynchronously, this
assertion is implemented asynchronously.  As such, it does not return
a boolean to indicate its passing status.  Instead, it returns a
Promise that resolves when it is completed.

## t.resolves(promise | fn, message, extra)

Verifies that the promise (or promise-returning function) resolves,
making no expectation about the value that the promise resolves to.

Note: since promises always reject and resolve asynchronously, this
assertion is implemented asynchronously.  As such, it does not return
a boolean to indicate its passing status.  Instead, it returns a
Promise that resolves when it is completed.

## t.resolveMatch(promise | fn, wanted, message, extra)

Verifies that the promise (or promise-returning function) resolves,
and furthermore that the value of the promise matches the `wanted`
pattern using `t.match`.

Note: since promises always reject and resolve asynchronously, this
assertion is implemented asynchronously.  As such, it does not return
a boolean to indicate its passing status.  Instead, it returns a
Promise that resolves when it is completed.

## t.resolveMatchSnapshot(promise | fn, message, extra)

Verifies that the promise (or promise-returning function) resolves,
and furthermore that the value of the promise matches the snapshot.

Note: since promises always reject and resolve asynchronously, this
assertion is implemented asynchronously.  As such, it does not return
a boolean to indicate its passing status.  Instead, it returns a
Promise that resolves when it is completed.

## t.throws(fn, [expectedError], message, extra)

Expect the function to throw an error.  If an expected error is
provided, then also verify that the thrown error matches the expected
error.

If the expected error is an object, then it's matched against the
thrown error using `t.match(er, expectedError)`.  If it's a function,
then the error is asserted to be a member of that class.

If the function has a name, and the message is not provided, then the
function name will be used as the message.

If the function is not provided, then this will be treated as a `todo`
test.

Caveat: if you pass a `extra` object to t.throws, then you MUST also
pass in an expected error, or else it will read the diag object as the
expected error, and get upset when your thrown error doesn't match
`{skip:true}` or whatever.

For example, this will not work as expected:

```javascript
// anti-example, do not use!
t.throws(function() {throw new Error('x')}, { skip: true })
```

But this is fine:

```javascript
// this example is ok to use.
// note the empty 'expected error' object.
// since it has no fields, it'll only verify that the thrown thing is
// an object, not the value of any properties
t.throws(function() {throw new Error('x')}, {}, { skip: true })
```

The expected error is tested against the throw error using `t.match`,
so regular expressions and the like are fine.  If the expected error
is an `Error` object, then the `stack` field is ignored, since that
will generally never match.

## t.doesNotThrow(fn, message, extra)

Verify that the provided function does not throw.

If the function has a name, and the message is not provided, then the
function name will be used as the message.

If the function is not provided, then this will be treated as a `todo`
test.

Note: if an error is encountered unexpectedly, it's often better to
simply throw it.  The Test object will handle this as a failure.

## t.expectUncaughtException([expectedError], [message], [extra])

Expect the function to throw an uncaught exception at some point in the
future, before the test ends.  If the test ends without having thrown the
expected error, then the test fails.

This is useful for verifying that an error thrown in some part of your code
will _not_ be handled, which would normally result in a program crash, and
verify behavior in those scenarios.  If the error is thrown synchronously,
or within a promise, then the `t.throws()` or `t.rejects()` methods are
usually more appropriate.

If called multiple times, then the uncaught exception errors must be
emitted in the order called.

Example:

```js
const t = require('tap')
t.test('sync throw', t => {
  t.plan(1)
  t.expectUncaughtException({ message: 'sync' })
  setImmediate(() => {
    throw new Error('sync')
  })
})

t.test('async throw', t => {
  t.plan(1)
  t.expectUncaughtException({ message: 'async' })
  setImmediate(async () => {
    throw new Error('async')
  })
})

t.test('multiple uncaughts must occur in the order specified', t => {
  t.plan(2)
  t.expectUncaughtException({ message: 'sync' })
  t.expectUncaughtException({ message: 'async' })
  process.nextTick(() => {
    throw new Error('sync')
  })
  setTimeout(async () => {
    throw new Error('async')
  }, 100)
})
```

**Note**: This method will _not_ properly link a thrown error to the
correct test object in some cases involving native modules on Node version
8, because the `async_hooks` module does not track the execution context ID
across native boundaries.

## t.equal(found, wanted, message, extra)

Verify that the object found is exactly the same (that is, `===`) to
the object that is wanted.

## t.not(found, notWanted, message, extra)

Inverse of `t.equal()`.

Verify that the object found is not exactly the same (that is, `!==`) as
the object that is wanted.

## t.same(found, wanted, message, extra)

Verify that the found object is deeply equivalent to the wanted
object.  Use non-strict equality for scalars (ie, `==`).  See:
[tcompare](http://npm.im/tcompare)

## t.notSame(found, notWanted, message, extra)

Inverse of `t.same()`.

Verify that the found object is not deeply equivalent to the
unwanted object.  Uses non-strict inequality (ie, `!=`) for scalars.

## t.strictSame(found, wanted, message, extra)

Strict version of `t.same()`.

Verify that the found object is deeply equivalent to the wanted
object.  Use strict equality for scalars (ie, `===`).

## t.strictNotSame(found, notWanted, message, extra)

Inverse of `t.strictSame()`.

Verify that the found object is not deeply equivalent to the unwanted
object.  Use strict equality for scalars (ie, `===`).

## t.hasStrict(found, pattern, message, extra)

Verify that the found object contains all of the provided fields, and that
they are of the same type and value as the pattern provided.

This does _not_ do advanced/loose matching based on constructor, regexp
patterns, and so on, like `t.match()` does.  You _may_ specify `key:
undefined` in the pattern to ensure that a field is not defined in the
found object, but it will not differentiate between a missing property and
a property set to `undefined`.

## t.notHasStrict(found, pattern, messsage, extra)

The inverse of `t.hasStrict()`.  Verifies that the found object does not
contain all of the provided fields, or if it does, that they are not the
same values and/or types.

## t.has(found, pattern, message, extra)

Verify that the found object contains all of the provided fields, and that
they coerce to the same values, even if the types do not match.

This does _not_ do advanced/loose matching based on constructor, regexp
patterns, and so on, like `t.match()` does.  You _may_ specify `key:
undefined` in the pattern to ensure that a field is not defined in the
found object, but it will not differentiate between a missing property and
a property set to `undefined`.

## t.notHas(found, pattern, message, extra)

The inverse of `t.has()`.  Verifies that the found object does not contain
all of the provided fields, or if it does, that they do not coerce to the
same values.

## t.hasProp(found, property, message, extra)

Verifies that the object `found` contains the property `property` and that
it is not `undefined`.  Searches the prototype chain as well as "own"
properties.

`property` must be a string, and `found` must be an object.

## t.hasProps(found, propertyList, message, extra)

Verifies that the object `found` contains each of the property names in
`propertyList`, and that they are not `undefined`.  Searches prototype
chain as well as "own" properties.

`propertyList` must be an iterable object (eg, an Array or Set) of property
names, and `found` must be an object.

## t.hasOwnProp(found, property, message, extra)

Verifies that the object `found` contains the property `property` and that
it is not `undefined`.  Does not search the prototype chain.

`property` must be a string, and `found` can be any value other than `null`
or `undefined` (though of course `false` and `true` have no ownProperties).

## t.hasOwnProps(found, propertyList, message, extra)

Verifies that the object `found` contains the property names in
`propertyList`, and that they are not `undefined`.  Does not search the
prototype chain.

`propertyList` must be an iterable object (eg, an Array or Set) of property
names, and `found` can be any value other than `null` or `undefined`
(though of course `false` and `true` have no ownProperties).

## t.has

## t.match(found, pattern, message, extra)

Verify that the found object matches the pattern provided.

If pattern is a regular expression, and found is a string, then verify
that the string matches the pattern.

If the pattern is a string, and found is a string, then verify that
the pattern occurs within the string somewhere.

If pattern is an object, then verify that all of the (enumerable)
fields in the pattern match the corresponding fields in the object
using this same algorithm.  For example, the pattern `{x:/a[sdf]{3}/}`
would successfully match `{x:'asdf',y:'z'}`.

This is useful when you want to verify that an object has a certain
set of required fields, but additional fields are ok.

See [tcompare](http://npm.im/tcompare) for the full details on how this
works.

## t.notMatch(found, pattern, message, extra)

Inverse of `match()`

Verify that the found object does not match the pattern provided.

## t.type(object, type, message, extra)

Verify that the object is of the type provided.

Type can be a string that matches the `typeof` value of the object, or
the string name of any constructor in the object's prototype chain, or
a constructor function in the object's prototype chain.

For example, all the following will pass:

```javascript
t.type(new Date(), 'object')
t.type(new Date(), 'Date')
t.type(new Date(), Date)
```
