'use strict';

require('mocha');
const assert = require('assert').strict;
const bashPath = require('bash-path');
const cp = require('child_process');
const braces = require('..');

const bash = input => {
  return cp.spawnSync(bashPath(), ['-c', `echo ${input}`])
    .stdout.toString()
    .split(/\s+/)
    .filter(Boolean);
};

const equal = (input, expected = bash(input), options) => {
  assert.deepEqual(braces.expand(input, options), expected);
};

/**
 * Bash 4.3 unit tests
 */

describe('bash', () => {
  const fixtures = [
    ['{1\\.2}', {}, ['{1.2}']],
    ['{1\\.2}', { keepEscaping: true }, ['{1\\.2}']],
    ['{"x,x"}', {}, ['{x,x}']],
    ['{x","x}', {}, ['{x,x}']],
    ['\'{x,x}\'', {}, ['{x,x}']],
    ['{x`,`x}', {}, ['{x,x}']],
    ['{x`,`x}', { keepQuotes: true }, ['{x`,`x}']],
    ['\'{a,b}{{a,b},a,b}\'', {}, ['{a,b}{{a,b},a,b}']],
    ['A{b,{d,e},{f,g}}Z', {}, ['AbZ', 'AdZ', 'AeZ', 'AfZ', 'AgZ']],
    ['PRE-{a,b}{{a,b},a,b}-POST', {}, ['PRE-aa-POST', 'PRE-ab-POST', 'PRE-aa-POST', 'PRE-ab-POST', 'PRE-ba-POST', 'PRE-bb-POST', 'PRE-ba-POST', 'PRE-bb-POST']],
    ['\\{a,b}{{a,b},a,b}', {}, ['{a,b}a', '{a,b}b', '{a,b}a', '{a,b}b']],
    ['{{a,b}', {}, ['{a', '{b']],
    ['{a,b}}', {}, ['a}', 'b}']],
    ['{,}', {}, ['', '']],
    ['a{,}', {}, ['a', 'a']],
    ['{,}b', {}, ['b', 'b']],
    ['a{,}b', {}, ['ab', 'ab']],
    ['a{b}c', {}, ['a{b}c']],
    ['a{1..5}b', {}, ['a1b', 'a2b', 'a3b', 'a4b', 'a5b']],
    ['a{01..5}b', {}, ['a01b', 'a02b', 'a03b', 'a04b', 'a05b']],
    ['a{-01..5}b', {}, ['a-01b', 'a000b', 'a001b', 'a002b', 'a003b', 'a004b', 'a005b']],
    ['a{-01..5..3}b', {}, ['a-01b', 'a002b', 'a005b']],
    ['a{001..9}b', {}, ['a001b', 'a002b', 'a003b', 'a004b', 'a005b', 'a006b', 'a007b', 'a008b', 'a009b']],
    ['a{b,c{d,e},{f,g}h}x{y,z', {}, ['abx{y,z', 'acdx{y,z', 'acex{y,z', 'afhx{y,z', 'aghx{y,z']],
    ['a{b,c{d,e},{f,g}h}x{y,z\\}', {}, ['abx{y,z}', 'acdx{y,z}', 'acex{y,z}', 'afhx{y,z}', 'aghx{y,z}']],
    ['a{b,c{d,e},{f,g}h}x{y,z}', {}, ['abxy', 'abxz', 'acdxy', 'acdxz', 'acexy', 'acexz', 'afhxy', 'afhxz', 'aghxy', 'aghxz']],
    ['a{b{c{d,e}f{x,y{{g}h', {}, ['a{b{cdf{x,y{{g}h', 'a{b{cef{x,y{{g}h']],
    ['a{b{c{d,e}f{x,y{}g}h', {}, ['a{b{cdfxh', 'a{b{cdfy{}gh', 'a{b{cefxh', 'a{b{cefy{}gh']],
    ['a{b{c{d,e}f{x,y}}g}h', {}, ['a{b{cdfx}g}h', 'a{b{cdfy}g}h', 'a{b{cefx}g}h', 'a{b{cefy}g}h']],
    ['a{b{c{d,e}f}g}h', {}, ['a{b{cdf}g}h', 'a{b{cef}g}h']],
    ['a{{x,y},z}b', {}, ['axb', 'ayb', 'azb']],
    ['f{x,y{g,z}}h', {}, ['fxh', 'fygh', 'fyzh']],
    ['f{x,y{{g,z}}h', {}, ['f{x,y{g}h', 'f{x,y{z}h']],
    ['f{x,y{{g,z}}h}', {}, ['fx', 'fy{g}h', 'fy{z}h']],
    ['f{x,y{{g}h', {}, ['f{x,y{{g}h']],
    ['f{x,y{{g}}h', {}, ['f{x,y{{g}}h']],
    ['f{x,y{}g}h', {}, ['fxh', 'fy{}gh']],
    ['z{a,b{,c}d', {}, ['z{a,bd', 'z{a,bcd']],
    ['z{a,b},c}d', {}, ['za,c}d', 'zb,c}d']],
    ['{-01..5}', {}, ['-01', '000', '001', '002', '003', '004', '005']],
    ['{-05..100..5}', {}, ['-05', '000', '005', '010', '015', '020', '025', '030', '035', '040', '045', '050', '055', '060', '065', '070', '075', '080', '085', '090', '095', '100']],
    ['{-05..100}', {}, ['-05', '-04', '-03', '-02', '-01', '000', '001', '002', '003', '004', '005', '006', '007', '008', '009', '010', '011', '012', '013', '014', '015', '016', '017', '018', '019', '020', '021', '022', '023', '024', '025', '026', '027', '028', '029', '030', '031', '032', '033', '034', '035', '036', '037', '038', '039', '040', '041', '042', '043', '044', '045', '046', '047', '048', '049', '050', '051', '052', '053', '054', '055', '056', '057', '058', '059', '060', '061', '062', '063', '064', '065', '066', '067', '068', '069', '070', '071', '072', '073', '074', '075', '076', '077', '078', '079', '080', '081', '082', '083', '084', '085', '086', '087', '088', '089', '090', '091', '092', '093', '094', '095', '096', '097', '098', '099', '100']],
    ['{0..5..2}', {}, ['0', '2', '4']],
    ['{0001..05..2}', {}, ['0001', '0003', '0005']],
    ['{0001..-5..2}', {}, ['0001', '-001', '-003', '-005']],
    ['{0001..-5..-2}', {}, ['0001', '-001', '-003', '-005']],
    ['{0001..5..-2}', {}, ['0001', '0003', '0005']],
    ['{01..5}', {}, ['01', '02', '03', '04', '05']],
    ['{1..05}', {}, ['01', '02', '03', '04', '05']],
    ['{1..05..3}', {}, ['01', '04']],
    ['{05..100}', {}, ['005', '006', '007', '008', '009', '010', '011', '012', '013', '014', '015', '016', '017', '018', '019', '020', '021', '022', '023', '024', '025', '026', '027', '028', '029', '030', '031', '032', '033', '034', '035', '036', '037', '038', '039', '040', '041', '042', '043', '044', '045', '046', '047', '048', '049', '050', '051', '052', '053', '054', '055', '056', '057', '058', '059', '060', '061', '062', '063', '064', '065', '066', '067', '068', '069', '070', '071', '072', '073', '074', '075', '076', '077', '078', '079', '080', '081', '082', '083', '084', '085', '086', '087', '088', '089', '090', '091', '092', '093', '094', '095', '096', '097', '098', '099', '100']],
    ['{0a..0z}', {}, ['{0a..0z}']],
    ['{a,b\\}c,d}', {}, ['a', 'b}c', 'd']],
    ['{a,b{c,d}', {}, ['{a,bc', '{a,bd']],
    ['{a,b}c,d}', {}, ['ac,d}', 'bc,d}']],
    ['{a..F}', {}, ['a', '`', '_', '^', ']', '\\', '[', 'Z', 'Y', 'X', 'W', 'V', 'U', 'T', 'S', 'R', 'Q', 'P', 'O', 'N', 'M', 'L', 'K', 'J', 'I', 'H', 'G', 'F']],
    ['{A..f}', {}, ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f']],
    ['{a..Z}', {}, ['a', '`', '_', '^', ']', '\\', '[', 'Z']],
    ['{A..z}', {}, ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z']],
    ['{z..A}', {}, ['z', 'y', 'x', 'w', 'v', 'u', 't', 's', 'r', 'q', 'p', 'o', 'n', 'm', 'l', 'k', 'j', 'i', 'h', 'g', 'f', 'e', 'd', 'c', 'b', 'a', '`', '_', '^', ']', '\\', '[', 'Z', 'Y', 'X', 'W', 'V', 'U', 'T', 'S', 'R', 'Q', 'P', 'O', 'N', 'M', 'L', 'K', 'J', 'I', 'H', 'G', 'F', 'E', 'D', 'C', 'B', 'A']],
    ['{Z..a}', {}, ['Z', '[', '\\', ']', '^', '_', '`', 'a']],
    ['{a..F..2}', {}, ['a', '_', ']', '[', 'Y', 'W', 'U', 'S', 'Q', 'O', 'M', 'K', 'I', 'G']],
    ['{A..f..02}', {}, ['A', 'C', 'E', 'G', 'I', 'K', 'M', 'O', 'Q', 'S', 'U', 'W', 'Y', '[', ']', '_', 'a', 'c', 'e']],
    ['{a..Z..5}', {}, ['a', '\\']],
    ['d{a..Z..5}b', {}, ['dab', 'd\\b']],
    ['{A..z..10}', {}, ['A', 'K', 'U', '_', 'i', 's']],
    ['{z..A..-2}', {}, ['z', 'x', 'v', 't', 'r', 'p', 'n', 'l', 'j', 'h', 'f', 'd', 'b', '`', '^', '\\', 'Z', 'X', 'V', 'T', 'R', 'P', 'N', 'L', 'J', 'H', 'F', 'D', 'B']],
    ['{Z..a..20}', {}, ['Z']],
    ['{a{,b}', {}, ['{a', '{ab']],
    ['{a\\},b}', {}, ['a}', 'b']],
    ['{x,y{,}g}', {}, ['x', 'yg', 'yg']],
    ['{x,y{}g}', {}, ['x', 'y{}g']],
    ['{{a,b}', {}, ['{a', '{b']],
    ['{{a,b},c}', {}, ['a', 'b', 'c']],
    ['{{a,b}c}', {}, ['{ac}', '{bc}']],
    ['{{a,b},}', {}, ['a', 'b', '']],
    ['X{{a,b},}X', {}, ['XaX', 'XbX', 'XX']],
    ['{{a,b},}c', {}, ['ac', 'bc', 'c']],
    ['{{a,b}.}', {}, ['{a.}', '{b.}']],
    ['{{a,b}}', {}, ['{a}', '{b}']],
    ['X{a..#}X', {}, ['XaX', 'X`X', 'X_X', 'X^X', 'X]X', 'X\\X', 'X[X', 'XZX', 'XYX', 'XXX', 'XWX', 'XVX', 'XUX', 'XTX', 'XSX', 'XRX', 'XQX', 'XPX', 'XOX', 'XNX', 'XMX', 'XLX', 'XKX', 'XJX', 'XIX', 'XHX', 'XGX', 'XFX', 'XEX', 'XDX', 'XCX', 'XBX', 'XAX', 'X@X', 'X?X', 'X>X', 'X=X', 'X<X', 'X;X', 'X:X', 'X9X', 'X8X', 'X7X', 'X6X', 'X5X', 'X4X', 'X3X', 'X2X', 'X1X', 'X0X', 'X/X', 'X.X', 'X-X', 'X,X', 'X+X', 'X*X', 'X)X', 'X(X', 'X\'X', 'X&X', 'X%X', 'X$X', 'X#X']],
    ['{-10..00}', {}, ['-10', '-09', '-08', '-07', '-06', '-05', '-04', '-03', '-02', '-01', '000']],
    ['{a,\\\\{a,b}c}', {}, ['a', '\\ac', '\\bc']],
    ['{a,\\{a,b}c}', {}, ['ac}', '{ac}', 'bc}']],
    ['a,\\{b,c}', {}, ['a,{b,c}']],
    ['{-10.\\.00}', {}, ['{-10..00}']],
    ['ff{c,b,a}', {}, ['ffc', 'ffb', 'ffa']],
    ['f{d,e,f}g', {}, ['fdg', 'feg', 'ffg']],
    ['{l,n,m}xyz', {}, ['lxyz', 'nxyz', 'mxyz']],
    ['{abc\\,def}', {}, ['{abc,def}']],
    ['{abc}', {}, ['{abc}']],
    ['{x\\,y,\\{abc\\},trie}', {}, ['x,y', '{abc}', 'trie']],
    ['{}', {}, ['{}']],
    ['{ }', {}, ['{ }']],
    ['}', {}, ['}']],
    ['{', {}, ['{']],
    ['abcd{efgh', {}, ['abcd{efgh']],
    ['foo {1,2} bar', {}, ['foo 1 bar', 'foo 2 bar']],
    ['"${var}"{x,y}', {}, ['${var}x', '${var}y']],
    ['{1..10}', {}, ['1', '2', '3', '4', '5', '6', '7', '8', '9', '10']],
    ['{0..10,braces}', {}, ['0..10', 'braces']],
    ['{{0..10},braces}', {}, ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '10', 'braces']],
    ['x{{0..10},braces}y', {}, ['x0y', 'x1y', 'x2y', 'x3y', 'x4y', 'x5y', 'x6y', 'x7y', 'x8y', 'x9y', 'x10y', 'xbracesy']],
    ['{3..3}', {}, ['3']],
    ['x{3..3}y', {}, ['x3y']],
    ['{10..1}', {}, ['10', '9', '8', '7', '6', '5', '4', '3', '2', '1']],
    ['{10..1}y', {}, ['10y', '9y', '8y', '7y', '6y', '5y', '4y', '3y', '2y', '1y']],
    ['x{10..1}y', {}, ['x10y', 'x9y', 'x8y', 'x7y', 'x6y', 'x5y', 'x4y', 'x3y', 'x2y', 'x1y']],
    ['{a..f}', {}, ['a', 'b', 'c', 'd', 'e', 'f']],
    ['{f..a}', {}, ['f', 'e', 'd', 'c', 'b', 'a']],
    ['{a..A}', {}, ['a', '`', '_', '^', ']', '\\', '[', 'Z', 'Y', 'X', 'W', 'V', 'U', 'T', 'S', 'R', 'Q', 'P', 'O', 'N', 'M', 'L', 'K', 'J', 'I', 'H', 'G', 'F', 'E', 'D', 'C', 'B', 'A']],
    ['{A..a}', {}, ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', '[', '\\', ']', '^', '_', '`', 'a']],
    ['{f..f}', {}, ['f']],
    ['0{1..9} {10..20}', {}, ['01 10', '01 11', '01 12', '01 13', '01 14', '01 15', '01 16', '01 17', '01 18', '01 19', '01 20', '02 10', '02 11', '02 12', '02 13', '02 14', '02 15', '02 16', '02 17', '02 18', '02 19', '02 20', '03 10', '03 11', '03 12', '03 13', '03 14', '03 15', '03 16', '03 17', '03 18', '03 19', '03 20', '04 10', '04 11', '04 12', '04 13', '04 14', '04 15', '04 16', '04 17', '04 18', '04 19', '04 20', '05 10', '05 11', '05 12', '05 13', '05 14', '05 15', '05 16', '05 17', '05 18', '05 19', '05 20', '06 10', '06 11', '06 12', '06 13', '06 14', '06 15', '06 16', '06 17', '06 18', '06 19', '06 20', '07 10', '07 11', '07 12', '07 13', '07 14', '07 15', '07 16', '07 17', '07 18', '07 19', '07 20', '08 10', '08 11', '08 12', '08 13', '08 14', '08 15', '08 16', '08 17', '08 18', '08 19', '08 20', '09 10', '09 11', '09 12', '09 13', '09 14', '09 15', '09 16', '09 17', '09 18', '09 19', '09 20']],
    ['{-1..-10}', {}, ['-1', '-2', '-3', '-4', '-5', '-6', '-7', '-8', '-9', '-10']],
    ['{-20..0}', {}, ['-20', '-19', '-18', '-17', '-16', '-15', '-14', '-13', '-12', '-11', '-10', '-9', '-8', '-7', '-6', '-5', '-4', '-3', '-2', '-1', '0']],
    ['a-{b{d,e}}-c', {}, ['a-{bd}-c', 'a-{be}-c']],
    ['a-{bdef-{g,i}-c', {}, ['a-{bdef-g-c', 'a-{bdef-i-c']],
    ['{"klklkl"}{1,2,3}', {}, ['{klklkl}1', '{klklkl}2', '{klklkl}3']],
    ['{"x,x"}', {}, ['{x,x}']],
    ['{klklkl}{1,2,3}', {}, ['{klklkl}1', '{klklkl}2', '{klklkl}3']],
    ['{1..10..2}', {}, ['1', '3', '5', '7', '9']],
    ['{-1..-10..2}', {}, ['-1', '-3', '-5', '-7', '-9']],
    ['{-1..-10..-2}', {}, ['-1', '-3', '-5', '-7', '-9']],
    ['{10..1..-2}', {}, ['10', '8', '6', '4', '2']],
    ['{10..1..2}', {}, ['10', '8', '6', '4', '2']],
    ['{1..20..2}', {}, ['1', '3', '5', '7', '9', '11', '13', '15', '17', '19']],
    ['{1..20..20}', {}, ['1']],
    ['{100..0..5}', {}, ['100', '95', '90', '85', '80', '75', '70', '65', '60', '55', '50', '45', '40', '35', '30', '25', '20', '15', '10', '5', '0']],
    ['{100..0..-5}', {}, ['100', '95', '90', '85', '80', '75', '70', '65', '60', '55', '50', '45', '40', '35', '30', '25', '20', '15', '10', '5', '0']],
    ['{a..z}', {}, ['a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z']],
    ['{a..z..2}', {}, ['a', 'c', 'e', 'g', 'i', 'k', 'm', 'o', 'q', 's', 'u', 'w', 'y']],
    ['{z..a..-2}', {}, ['z', 'x', 'v', 't', 'r', 'p', 'n', 'l', 'j', 'h', 'f', 'd', 'b']],
    ['{2147483645..2147483649}', {}, ['2147483645', '2147483646', '2147483647', '2147483648', '2147483649']],
    ['{10..0..2}', {}, ['10', '8', '6', '4', '2', '0']],
    ['{10..0..-2}', {}, ['10', '8', '6', '4', '2', '0']],
    ['{-50..-0..5}', {}, ['-50', '-45', '-40', '-35', '-30', '-25', '-20', '-15', '-10', '-5', '0']],
    ['{1..10.f}', {}, ['{1..10.f}']],
    ['{1..ff}', {}, ['{1..ff}']],
    ['{1..10..ff}', {}, ['{1..10..ff}']],
    ['{1.20..2}', {}, ['{1.20..2}']],
    ['{1..20..f2}', {}, ['{1..20..f2}']],
    ['{1..20..2f}', {}, ['{1..20..2f}']],
    ['{1..2f..2}', {}, ['{1..2f..2}']],
    ['{1..ff..2}', {}, ['{1..ff..2}']],
    ['{1..ff}', {}, ['{1..ff}']],
    ['{1..0f}', {}, ['{1..0f}']],
    ['{1..10f}', {}, ['{1..10f}']],
    ['{1..10.f}', {}, ['{1..10.f}']],
    ['{},b}.h', {}, ['{},b}.h']],
    ['y{\\},a}x', {}, ['y}x', 'yax']],
    ['{}a,b}c', {}, ['{}a,b}c']]
  ];

  fixtures.forEach(arr => {
    if (typeof arr === 'string') {
      return;
    }

    const options = { ...arr[1] };
    const pattern = arr[0];
    const expected = arr[2];

    if (options.skip === true) {
      return;
    }

    it('should compile: ' + pattern, () => {
      equal(pattern, expected, options);
    });
  });
});
