/*-------------------------------------------------------------------------
 *
 * pg_crc32c_sb8.c
 *	  Compute CRC-32C checksum using slicing-by-8 algorithm.
 *
 * Michael E. Kounavis, Frank L. Berry,
 * "Novel Table Lookup-Based Algorithms for High-Performance CRC
 * Generation", IEEE Transactions on Computers, vol.57, no. 11,
 * pp. 1550-1560, November 2008, doi:10.1109/TC.2008.85
 *
 * Portions Copyright (c) 1996-2022, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  src/port/pg_crc32c_sb8.c
 *
 *-------------------------------------------------------------------------
 */

#include "c.h"

#include "port/pg_crc32c.h"

static const uint32 pg_crc32c_table[8][256];

/* Accumulate one input byte */
#ifdef WORDS_BIGENDIAN
#define CRC8(x) pg_crc32c_table[0][((crc >> 24) ^ (x)) & 0xFF] ^ (crc << 8)
#else
#define CRC8(x) pg_crc32c_table[0][(crc ^ (x)) & 0xFF] ^ (crc >> 8)
#endif

pg_crc32c
pg_comp_crc32c_sb8(pg_crc32c crc, const void *data, size_t len)
{
	const unsigned char *p = data;
	const uint32 *p4;

	/*
	 * Handle 0-3 initial bytes one at a time, so that the loop below starts
	 * with a pointer aligned to four bytes.
	 */
	while (len > 0 && ((uintptr_t) p & 3))
	{
		crc = CRC8(*p++);
		len--;
	}

	/*
	 * Process eight bytes of data at a time.
	 */
	p4 = (const uint32 *) p;
	while (len >= 8)
	{
		uint32		a = *p4++ ^ crc;
		uint32		b = *p4++;

#ifdef WORDS_BIGENDIAN
		const uint8 c0 = b;
		const uint8 c1 = b >> 8;
		const uint8 c2 = b >> 16;
		const uint8 c3 = b >> 24;
		const uint8 c4 = a;
		const uint8 c5 = a >> 8;
		const uint8 c6 = a >> 16;
		const uint8 c7 = a >> 24;
#else
		const uint8 c0 = b >> 24;
		const uint8 c1 = b >> 16;
		const uint8 c2 = b >> 8;
		const uint8 c3 = b;
		const uint8 c4 = a >> 24;
		const uint8 c5 = a >> 16;
		const uint8 c6 = a >> 8;
		const uint8 c7 = a;
#endif

		crc =
			pg_crc32c_table[0][c0] ^ pg_crc32c_table[1][c1] ^
			pg_crc32c_table[2][c2] ^ pg_crc32c_table[3][c3] ^
			pg_crc32c_table[4][c4] ^ pg_crc32c_table[5][c5] ^
			pg_crc32c_table[6][c6] ^ pg_crc32c_table[7][c7];

		len -= 8;
	}

	/*
	 * Handle any remaining bytes one at a time.
	 */
	p = (const unsigned char *) p4;
	while (len > 0)
	{
		crc = CRC8(*p++);
		len--;
	}

	return crc;
}

/*
 * Lookup tables for the slicing-by-8 algorithm, for the so-called Castagnoli
 * polynomial (the same that is used e.g. in iSCSI), 0x1EDC6F41. Using
 * Williams' terms, this is the "normal", not "reflected" version. However, on
 * big-endian systems the values in the tables are stored in byte-reversed
 * order (IOW, the tables are stored in little-endian order even on big-endian
 * systems).
 */
static const uint32 pg_crc32c_table[8][256] = {
#ifndef WORDS_BIGENDIAN
	{
		0x00000000, 0xF26B8303, 0xE13B70F7, 0x1350F3F4,
		0xC79A971F, 0x35F1141C, 0x26A1E7E8, 0xD4CA64EB,
		0x8AD958CF, 0x78B2DBCC, 0x6BE22838, 0x9989AB3B,
		0x4D43CFD0, 0xBF284CD3, 0xAC78BF27, 0x5E133C24,
		0x105EC76F, 0xE235446C, 0xF165B798, 0x030E349B,
		0xD7C45070, 0x25AFD373, 0x36FF2087, 0xC494A384,
		0x9A879FA0, 0x68EC1CA3, 0x7BBCEF57, 0x89D76C54,
		0x5D1D08BF, 0xAF768BBC, 0xBC267848, 0x4E4DFB4B,
		0x20BD8EDE, 0xD2D60DDD, 0xC186FE29, 0x33ED7D2A,
		0xE72719C1, 0x154C9AC2, 0x061C6936, 0xF477EA35,
		0xAA64D611, 0x580F5512, 0x4B5FA6E6, 0xB93425E5,
		0x6DFE410E, 0x9F95C20D, 0x8CC531F9, 0x7EAEB2FA,
		0x30E349B1, 0xC288CAB2, 0xD1D83946, 0x23B3BA45,
		0xF779DEAE, 0x05125DAD, 0x1642AE59, 0xE4292D5A,
		0xBA3A117E, 0x4851927D, 0x5B016189, 0xA96AE28A,
		0x7DA08661, 0x8FCB0562, 0x9C9BF696, 0x6EF07595,
		0x417B1DBC, 0xB3109EBF, 0xA0406D4B, 0x522BEE48,
		0x86E18AA3, 0x748A09A0, 0x67DAFA54, 0x95B17957,
		0xCBA24573, 0x39C9C670, 0x2A993584, 0xD8F2B687,
		0x0C38D26C, 0xFE53516F, 0xED03A29B, 0x1F682198,
		0x5125DAD3, 0xA34E59D0, 0xB01EAA24, 0x42752927,
		0x96BF4DCC, 0x64D4CECF, 0x77843D3B, 0x85EFBE38,
		0xDBFC821C, 0x2997011F, 0x3AC7F2EB, 0xC8AC71E8,
		0x1C661503, 0xEE0D9600, 0xFD5D65F4, 0x0F36E6F7,
		0x61C69362, 0x93AD1061, 0x80FDE395, 0x72966096,
		0xA65C047D, 0x5437877E, 0x4767748A, 0xB50CF789,
		0xEB1FCBAD, 0x197448AE, 0x0A24BB5A, 0xF84F3859,
		0x2C855CB2, 0xDEEEDFB1, 0xCDBE2C45, 0x3FD5AF46,
		0x7198540D, 0x83F3D70E, 0x90A324FA, 0x62C8A7F9,
		0xB602C312, 0x44694011, 0x5739B3E5, 0xA55230E6,
		0xFB410CC2, 0x092A8FC1, 0x1A7A7C35, 0xE811FF36,
		0x3CDB9BDD, 0xCEB018DE, 0xDDE0EB2A, 0x2F8B6829,
		0x82F63B78, 0x709DB87B, 0x63CD4B8F, 0x91A6C88C,
		0x456CAC67, 0xB7072F64, 0xA457DC90, 0x563C5F93,
		0x082F63B7, 0xFA44E0B4, 0xE9141340, 0x1B7F9043,
		0xCFB5F4A8, 0x3DDE77AB, 0x2E8E845F, 0xDCE5075C,
		0x92A8FC17, 0x60C37F14, 0x73938CE0, 0x81F80FE3,
		0x55326B08, 0xA759E80B, 0xB4091BFF, 0x466298FC,
		0x1871A4D8, 0xEA1A27DB, 0xF94AD42F, 0x0B21572C,
		0xDFEB33C7, 0x2D80B0C4, 0x3ED04330, 0xCCBBC033,
		0xA24BB5A6, 0x502036A5, 0x4370C551, 0xB11B4652,
		0x65D122B9, 0x97BAA1BA, 0x84EA524E, 0x7681D14D,
		0x2892ED69, 0xDAF96E6A, 0xC9A99D9E, 0x3BC21E9D,
		0xEF087A76, 0x1D63F975, 0x0E330A81, 0xFC588982,
		0xB21572C9, 0x407EF1CA, 0x532E023E, 0xA145813D,
		0x758FE5D6, 0x87E466D5, 0x94B49521, 0x66DF1622,
		0x38CC2A06, 0xCAA7A905, 0xD9F75AF1, 0x2B9CD9F2,
		0xFF56BD19, 0x0D3D3E1A, 0x1E6DCDEE, 0xEC064EED,
		0xC38D26C4, 0x31E6A5C7, 0x22B65633, 0xD0DDD530,
		0x0417B1DB, 0xF67C32D8, 0xE52CC12C, 0x1747422F,
		0x49547E0B, 0xBB3FFD08, 0xA86F0EFC, 0x5A048DFF,
		0x8ECEE914, 0x7CA56A17, 0x6FF599E3, 0x9D9E1AE0,
		0xD3D3E1AB, 0x21B862A8, 0x32E8915C, 0xC083125F,
		0x144976B4, 0xE622F5B7, 0xF5720643, 0x07198540,
		0x590AB964, 0xAB613A67, 0xB831C993, 0x4A5A4A90,
		0x9E902E7B, 0x6CFBAD78, 0x7FAB5E8C, 0x8DC0DD8F,
		0xE330A81A, 0x115B2B19, 0x020BD8ED, 0xF0605BEE,
		0x24AA3F05, 0xD6C1BC06, 0xC5914FF2, 0x37FACCF1,
		0x69E9F0D5, 0x9B8273D6, 0x88D28022, 0x7AB90321,
		0xAE7367CA, 0x5C18E4C9, 0x4F48173D, 0xBD23943E,
		0xF36E6F75, 0x0105EC76, 0x12551F82, 0xE03E9C81,
		0x34F4F86A, 0xC69F7B69, 0xD5CF889D, 0x27A40B9E,
		0x79B737BA, 0x8BDCB4B9, 0x988C474D, 0x6AE7C44E,
		0xBE2DA0A5, 0x4C4623A6, 0x5F16D052, 0xAD7D5351
	},
	{
		0x00000000, 0x13A29877, 0x274530EE, 0x34E7A899,
		0x4E8A61DC, 0x5D28F9AB, 0x69CF5132, 0x7A6DC945,
		0x9D14C3B8, 0x8EB65BCF, 0xBA51F356, 0xA9F36B21,
		0xD39EA264, 0xC03C3A13, 0xF4DB928A, 0xE7790AFD,
		0x3FC5F181, 0x2C6769F6, 0x1880C16F, 0x0B225918,
		0x714F905D, 0x62ED082A, 0x560AA0B3, 0x45A838C4,
		0xA2D13239, 0xB173AA4E, 0x859402D7, 0x96369AA0,
		0xEC5B53E5, 0xFFF9CB92, 0xCB1E630B, 0xD8BCFB7C,
		0x7F8BE302, 0x6C297B75, 0x58CED3EC, 0x4B6C4B9B,
		0x310182DE, 0x22A31AA9, 0x1644B230, 0x05E62A47,
		0xE29F20BA, 0xF13DB8CD, 0xC5DA1054, 0xD6788823,
		0xAC154166, 0xBFB7D911, 0x8B507188, 0x98F2E9FF,
		0x404E1283, 0x53EC8AF4, 0x670B226D, 0x74A9BA1A,
		0x0EC4735F, 0x1D66EB28, 0x298143B1, 0x3A23DBC6,
		0xDD5AD13B, 0xCEF8494C, 0xFA1FE1D5, 0xE9BD79A2,
		0x93D0B0E7, 0x80722890, 0xB4958009, 0xA737187E,
		0xFF17C604, 0xECB55E73, 0xD852F6EA, 0xCBF06E9D,
		0xB19DA7D8, 0xA23F3FAF, 0x96D89736, 0x857A0F41,
		0x620305BC, 0x71A19DCB, 0x45463552, 0x56E4AD25,
		0x2C896460, 0x3F2BFC17, 0x0BCC548E, 0x186ECCF9,
		0xC0D23785, 0xD370AFF2, 0xE797076B, 0xF4359F1C,
		0x8E585659, 0x9DFACE2E, 0xA91D66B7, 0xBABFFEC0,
		0x5DC6F43D, 0x4E646C4A, 0x7A83C4D3, 0x69215CA4,
		0x134C95E1, 0x00EE0D96, 0x3409A50F, 0x27AB3D78,
		0x809C2506, 0x933EBD71, 0xA7D915E8, 0xB47B8D9F,
		0xCE1644DA, 0xDDB4DCAD, 0xE9537434, 0xFAF1EC43,
		0x1D88E6BE, 0x0E2A7EC9, 0x3ACDD650, 0x296F4E27,
		0x53028762, 0x40A01F15, 0x7447B78C, 0x67E52FFB,
		0xBF59D487, 0xACFB4CF0, 0x981CE469, 0x8BBE7C1E,
		0xF1D3B55B, 0xE2712D2C, 0xD69685B5, 0xC5341DC2,
		0x224D173F, 0x31EF8F48, 0x050827D1, 0x16AABFA6,
		0x6CC776E3, 0x7F65EE94, 0x4B82460D, 0x5820DE7A,
		0xFBC3FAF9, 0xE861628E, 0xDC86CA17, 0xCF245260,
		0xB5499B25, 0xA6EB0352, 0x920CABCB, 0x81AE33BC,
		0x66D73941, 0x7575A136, 0x419209AF, 0x523091D8,
		0x285D589D, 0x3BFFC0EA, 0x0F186873, 0x1CBAF004,
		0xC4060B78, 0xD7A4930F, 0xE3433B96, 0xF0E1A3E1,
		0x8A8C6AA4, 0x992EF2D3, 0xADC95A4A, 0xBE6BC23D,
		0x5912C8C0, 0x4AB050B7, 0x7E57F82E, 0x6DF56059,
		0x1798A91C, 0x043A316B, 0x30DD99F2, 0x237F0185,
		0x844819FB, 0x97EA818C, 0xA30D2915, 0xB0AFB162,
		0xCAC27827, 0xD960E050, 0xED8748C9, 0xFE25D0BE,
		0x195CDA43, 0x0AFE4234, 0x3E19EAAD, 0x2DBB72DA,
		0x57D6BB9F, 0x447423E8, 0x70938B71, 0x63311306,
		0xBB8DE87A, 0xA82F700D, 0x9CC8D894, 0x8F6A40E3,
		0xF50789A6, 0xE6A511D1, 0xD242B948, 0xC1E0213F,
		0x26992BC2, 0x353BB3B5, 0x01DC1B2C, 0x127E835B,
		0x68134A1E, 0x7BB1D269, 0x4F567AF0, 0x5CF4E287,
		0x04D43CFD, 0x1776A48A, 0x23910C13, 0x30339464,
		0x4A5E5D21, 0x59FCC556, 0x6D1B6DCF, 0x7EB9F5B8,
		0x99C0FF45, 0x8A626732, 0xBE85CFAB, 0xAD2757DC,
		0xD74A9E99, 0xC4E806EE, 0xF00FAE77, 0xE3AD3600,
		0x3B11CD7C, 0x28B3550B, 0x1C54FD92, 0x0FF665E5,
		0x759BACA0, 0x663934D7, 0x52DE9C4E, 0x417C0439,
		0xA6050EC4, 0xB5A796B3, 0x81403E2A, 0x92E2A65D,
		0xE88F6F18, 0xFB2DF76F, 0xCFCA5FF6, 0xDC68C781,
		0x7B5FDFFF, 0x68FD4788, 0x5C1AEF11, 0x4FB87766,
		0x35D5BE23, 0x26772654, 0x12908ECD, 0x013216BA,
		0xE64B1C47, 0xF5E98430, 0xC10E2CA9, 0xD2ACB4DE,
		0xA8C17D9B, 0xBB63E5EC, 0x8F844D75, 0x9C26D502,
		0x449A2E7E, 0x5738B609, 0x63DF1E90, 0x707D86E7,
		0x0A104FA2, 0x19B2D7D5, 0x2D557F4C, 0x3EF7E73B,
		0xD98EEDC6, 0xCA2C75B1, 0xFECBDD28, 0xED69455F,
		0x97048C1A, 0x84A6146D, 0xB041BCF4, 0xA3E32483
	},
	{
		0x00000000, 0xA541927E, 0x4F6F520D, 0xEA2EC073,
		0x9EDEA41A, 0x3B9F3664, 0xD1B1F617, 0x74F06469,
		0x38513EC5, 0x9D10ACBB, 0x773E6CC8, 0xD27FFEB6,
		0xA68F9ADF, 0x03CE08A1, 0xE9E0C8D2, 0x4CA15AAC,
		0x70A27D8A, 0xD5E3EFF4, 0x3FCD2F87, 0x9A8CBDF9,
		0xEE7CD990, 0x4B3D4BEE, 0xA1138B9D, 0x045219E3,
		0x48F3434F, 0xEDB2D131, 0x079C1142, 0xA2DD833C,
		0xD62DE755, 0x736C752B, 0x9942B558, 0x3C032726,
		0xE144FB14, 0x4405696A, 0xAE2BA919, 0x0B6A3B67,
		0x7F9A5F0E, 0xDADBCD70, 0x30F50D03, 0x95B49F7D,
		0xD915C5D1, 0x7C5457AF, 0x967A97DC, 0x333B05A2,
		0x47CB61CB, 0xE28AF3B5, 0x08A433C6, 0xADE5A1B8,
		0x91E6869E, 0x34A714E0, 0xDE89D493, 0x7BC846ED,
		0x0F382284, 0xAA79B0FA, 0x40577089, 0xE516E2F7,
		0xA9B7B85B, 0x0CF62A25, 0xE6D8EA56, 0x43997828,
		0x37691C41, 0x92288E3F, 0x78064E4C, 0xDD47DC32,
		0xC76580D9, 0x622412A7, 0x880AD2D4, 0x2D4B40AA,
		0x59BB24C3, 0xFCFAB6BD, 0x16D476CE, 0xB395E4B0,
		0xFF34BE1C, 0x5A752C62, 0xB05BEC11, 0x151A7E6F,
		0x61EA1A06, 0xC4AB8878, 0x2E85480B, 0x8BC4DA75,
		0xB7C7FD53, 0x12866F2D, 0xF8A8AF5E, 0x5DE93D20,
		0x29195949, 0x8C58CB37, 0x66760B44, 0xC337993A,
		0x8F96C396, 0x2AD751E8, 0xC0F9919B, 0x65B803E5,
		0x1148678C, 0xB409F5F2, 0x5E273581, 0xFB66A7FF,
		0x26217BCD, 0x8360E9B3, 0x694E29C0, 0xCC0FBBBE,
		0xB8FFDFD7, 0x1DBE4DA9, 0xF7908DDA, 0x52D11FA4,
		0x1E704508, 0xBB31D776, 0x511F1705, 0xF45E857B,
		0x80AEE112, 0x25EF736C, 0xCFC1B31F, 0x6A802161,
		0x56830647, 0xF3C29439, 0x19EC544A, 0xBCADC634,
		0xC85DA25D, 0x6D1C3023, 0x8732F050, 0x2273622E,
		0x6ED23882, 0xCB93AAFC, 0x21BD6A8F, 0x84FCF8F1,
		0xF00C9C98, 0x554D0EE6, 0xBF63CE95, 0x1A225CEB,
		0x8B277743, 0x2E66E53D, 0xC448254E, 0x6109B730,
		0x15F9D359, 0xB0B84127, 0x5A968154, 0xFFD7132A,
		0xB3764986, 0x1637DBF8, 0xFC191B8B, 0x595889F5,
		0x2DA8ED9C, 0x88E97FE2, 0x62C7BF91, 0xC7862DEF,
		0xFB850AC9, 0x5EC498B7, 0xB4EA58C4, 0x11ABCABA,
		0x655BAED3, 0xC01A3CAD, 0x2A34FCDE, 0x8F756EA0,
		0xC3D4340C, 0x6695A672, 0x8CBB6601, 0x29FAF47F,
		0x5D0A9016, 0xF84B0268, 0x1265C21B, 0xB7245065,
		0x6A638C57, 0xCF221E29, 0x250CDE5A, 0x804D4C24,
		0xF4BD284D, 0x51FCBA33, 0xBBD27A40, 0x1E93E83E,
		0x5232B292, 0xF77320EC, 0x1D5DE09F, 0xB81C72E1,
		0xCCEC1688, 0x69AD84F6, 0x83834485, 0x26C2D6FB,
		0x1AC1F1DD, 0xBF8063A3, 0x55AEA3D0, 0xF0EF31AE,
		0x841F55C7, 0x215EC7B9, 0xCB7007CA, 0x6E3195B4,
		0x2290CF18, 0x87D15D66, 0x6DFF9D15, 0xC8BE0F6B,
		0xBC4E6B02, 0x190FF97C, 0xF321390F, 0x5660AB71,
		0x4C42F79A, 0xE90365E4, 0x032DA597, 0xA66C37E9,
		0xD29C5380, 0x77DDC1FE, 0x9DF3018D, 0x38B293F3,
		0x7413C95F, 0xD1525B21, 0x3B7C9B52, 0x9E3D092C,
		0xEACD6D45, 0x4F8CFF3B, 0xA5A23F48, 0x00E3AD36,
		0x3CE08A10, 0x99A1186E, 0x738FD81D, 0xD6CE4A63,
		0xA23E2E0A, 0x077FBC74, 0xED517C07, 0x4810EE79,
		0x04B1B4D5, 0xA1F026AB, 0x4BDEE6D8, 0xEE9F74A6,
		0x9A6F10CF, 0x3F2E82B1, 0xD50042C2, 0x7041D0BC,
		0xAD060C8E, 0x08479EF0, 0xE2695E83, 0x4728CCFD,
		0x33D8A894, 0x96993AEA, 0x7CB7FA99, 0xD9F668E7,
		0x9557324B, 0x3016A035, 0xDA386046, 0x7F79F238,
		0x0B899651, 0xAEC8042F, 0x44E6C45C, 0xE1A75622,
		0xDDA47104, 0x78E5E37A, 0x92CB2309, 0x378AB177,
		0x437AD51E, 0xE63B4760, 0x0C158713, 0xA954156D,
		0xE5F54FC1, 0x40B4DDBF, 0xAA9A1DCC, 0x0FDB8FB2,
		0x7B2BEBDB, 0xDE6A79A5, 0x3444B9D6, 0x91052BA8
	},
	{
		0x00000000, 0xDD45AAB8, 0xBF672381, 0x62228939,
		0x7B2231F3, 0xA6679B4B, 0xC4451272, 0x1900B8CA,
		0xF64463E6, 0x2B01C95E, 0x49234067, 0x9466EADF,
		0x8D665215, 0x5023F8AD, 0x32017194, 0xEF44DB2C,
		0xE964B13D, 0x34211B85, 0x560392BC, 0x8B463804,
		0x924680CE, 0x4F032A76, 0x2D21A34F, 0xF06409F7,
		0x1F20D2DB, 0xC2657863, 0xA047F15A, 0x7D025BE2,
		0x6402E328, 0xB9474990, 0xDB65C0A9, 0x06206A11,
		0xD725148B, 0x0A60BE33, 0x6842370A, 0xB5079DB2,
		0xAC072578, 0x71428FC0, 0x136006F9, 0xCE25AC41,
		0x2161776D, 0xFC24DDD5, 0x9E0654EC, 0x4343FE54,
		0x5A43469E, 0x8706EC26, 0xE524651F, 0x3861CFA7,
		0x3E41A5B6, 0xE3040F0E, 0x81268637, 0x5C632C8F,
		0x45639445, 0x98263EFD, 0xFA04B7C4, 0x27411D7C,
		0xC805C650, 0x15406CE8, 0x7762E5D1, 0xAA274F69,
		0xB327F7A3, 0x6E625D1B, 0x0C40D422, 0xD1057E9A,
		0xABA65FE7, 0x76E3F55F, 0x14C17C66, 0xC984D6DE,
		0xD0846E14, 0x0DC1C4AC, 0x6FE34D95, 0xB2A6E72D,
		0x5DE23C01, 0x80A796B9, 0xE2851F80, 0x3FC0B538,
		0x26C00DF2, 0xFB85A74A, 0x99A72E73, 0x44E284CB,
		0x42C2EEDA, 0x9F874462, 0xFDA5CD5B, 0x20E067E3,
		0x39E0DF29, 0xE4A57591, 0x8687FCA8, 0x5BC25610,
		0xB4868D3C, 0x69C32784, 0x0BE1AEBD, 0xD6A40405,
		0xCFA4BCCF, 0x12E11677, 0x70C39F4E, 0xAD8635F6,
		0x7C834B6C, 0xA1C6E1D4, 0xC3E468ED, 0x1EA1C255,
		0x07A17A9F, 0xDAE4D027, 0xB8C6591E, 0x6583F3A6,
		0x8AC7288A, 0x57828232, 0x35A00B0B, 0xE8E5A1B3,
		0xF1E51979, 0x2CA0B3C1, 0x4E823AF8, 0x93C79040,
		0x95E7FA51, 0x48A250E9, 0x2A80D9D0, 0xF7C57368,
		0xEEC5CBA2, 0x3380611A, 0x51A2E823, 0x8CE7429B,
		0x63A399B7, 0xBEE6330F, 0xDCC4BA36, 0x0181108E,
		0x1881A844, 0xC5C402FC, 0xA7E68BC5, 0x7AA3217D,
		0x52A0C93F, 0x8FE56387, 0xEDC7EABE, 0x30824006,
		0x2982F8CC, 0xF4C75274, 0x96E5DB4D, 0x4BA071F5,
		0xA4E4AAD9, 0x79A10061, 0x1B838958, 0xC6C623E0,
		0xDFC69B2A, 0x02833192, 0x60A1B8AB, 0xBDE41213,
		0xBBC47802, 0x6681D2BA, 0x04A35B83, 0xD9E6F13B,
		0xC0E649F1, 0x1DA3E349, 0x7F816A70, 0xA2C4C0C8,
		0x4D801BE4, 0x90C5B15C, 0xF2E73865, 0x2FA292DD,
		0x36A22A17, 0xEBE780AF, 0x89C50996, 0x5480A32E,
		0x8585DDB4, 0x58C0770C, 0x3AE2FE35, 0xE7A7548D,
		0xFEA7EC47, 0x23E246FF, 0x41C0CFC6, 0x9C85657E,
		0x73C1BE52, 0xAE8414EA, 0xCCA69DD3, 0x11E3376B,
		0x08E38FA1, 0xD5A62519, 0xB784AC20, 0x6AC10698,
		0x6CE16C89, 0xB1A4C631, 0xD3864F08, 0x0EC3E5B0,
		0x17C35D7A, 0xCA86F7C2, 0xA8A47EFB, 0x75E1D443,
		0x9AA50F6F, 0x47E0A5D7, 0x25C22CEE, 0xF8878656,
		0xE1873E9C, 0x3CC29424, 0x5EE01D1D, 0x83A5B7A5,
		0xF90696D8, 0x24433C60, 0x4661B559, 0x9B241FE1,
		0x8224A72B, 0x5F610D93, 0x3D4384AA, 0xE0062E12,
		0x0F42F53E, 0xD2075F86, 0xB025D6BF, 0x6D607C07,
		0x7460C4CD, 0xA9256E75, 0xCB07E74C, 0x16424DF4,
		0x106227E5, 0xCD278D5D, 0xAF050464, 0x7240AEDC,
		0x6B401616, 0xB605BCAE, 0xD4273597, 0x09629F2F,
		0xE6264403, 0x3B63EEBB, 0x59416782, 0x8404CD3A,
		0x9D0475F0, 0x4041DF48, 0x22635671, 0xFF26FCC9,
		0x2E238253, 0xF36628EB, 0x9144A1D2, 0x4C010B6A,
		0x5501B3A0, 0x88441918, 0xEA669021, 0x37233A99,
		0xD867E1B5, 0x05224B0D, 0x6700C234, 0xBA45688C,
		0xA345D046, 0x7E007AFE, 0x1C22F3C7, 0xC167597F,
		0xC747336E, 0x1A0299D6, 0x782010EF, 0xA565BA57,
		0xBC65029D, 0x6120A825, 0x0302211C, 0xDE478BA4,
		0x31035088, 0xEC46FA30, 0x8E647309, 0x5321D9B1,
		0x4A21617B, 0x9764CBC3, 0xF54642FA, 0x2803E842
	},
	{
		0x00000000, 0x38116FAC, 0x7022DF58, 0x4833B0F4,
		0xE045BEB0, 0xD854D11C, 0x906761E8, 0xA8760E44,
		0xC5670B91, 0xFD76643D, 0xB545D4C9, 0x8D54BB65,
		0x2522B521, 0x1D33DA8D, 0x55006A79, 0x6D1105D5,
		0x8F2261D3, 0xB7330E7F, 0xFF00BE8B, 0xC711D127,
		0x6F67DF63, 0x5776B0CF, 0x1F45003B, 0x27546F97,
		0x4A456A42, 0x725405EE, 0x3A67B51A, 0x0276DAB6,
		0xAA00D4F2, 0x9211BB5E, 0xDA220BAA, 0xE2336406,
		0x1BA8B557, 0x23B9DAFB, 0x6B8A6A0F, 0x539B05A3,
		0xFBED0BE7, 0xC3FC644B, 0x8BCFD4BF, 0xB3DEBB13,
		0xDECFBEC6, 0xE6DED16A, 0xAEED619E, 0x96FC0E32,
		0x3E8A0076, 0x069B6FDA, 0x4EA8DF2E, 0x76B9B082,
		0x948AD484, 0xAC9BBB28, 0xE4A80BDC, 0xDCB96470,
		0x74CF6A34, 0x4CDE0598, 0x04EDB56C, 0x3CFCDAC0,
		0x51EDDF15, 0x69FCB0B9, 0x21CF004D, 0x19DE6FE1,
		0xB1A861A5, 0x89B90E09, 0xC18ABEFD, 0xF99BD151,
		0x37516AAE, 0x0F400502, 0x4773B5F6, 0x7F62DA5A,
		0xD714D41E, 0xEF05BBB2, 0xA7360B46, 0x9F2764EA,
		0xF236613F, 0xCA270E93, 0x8214BE67, 0xBA05D1CB,
		0x1273DF8F, 0x2A62B023, 0x625100D7, 0x5A406F7B,
		0xB8730B7D, 0x806264D1, 0xC851D425, 0xF040BB89,
		0x5836B5CD, 0x6027DA61, 0x28146A95, 0x10050539,
		0x7D1400EC, 0x45056F40, 0x0D36DFB4, 0x3527B018,
		0x9D51BE5C, 0xA540D1F0, 0xED736104, 0xD5620EA8,
		0x2CF9DFF9, 0x14E8B055, 0x5CDB00A1, 0x64CA6F0D,
		0xCCBC6149, 0xF4AD0EE5, 0xBC9EBE11, 0x848FD1BD,
		0xE99ED468, 0xD18FBBC4, 0x99BC0B30, 0xA1AD649C,
		0x09DB6AD8, 0x31CA0574, 0x79F9B580, 0x41E8DA2C,
		0xA3DBBE2A, 0x9BCAD186, 0xD3F96172, 0xEBE80EDE,
		0x439E009A, 0x7B8F6F36, 0x33BCDFC2, 0x0BADB06E,
		0x66BCB5BB, 0x5EADDA17, 0x169E6AE3, 0x2E8F054F,
		0x86F90B0B, 0xBEE864A7, 0xF6DBD453, 0xCECABBFF,
		0x6EA2D55C, 0x56B3BAF0, 0x1E800A04, 0x269165A8,
		0x8EE76BEC, 0xB6F60440, 0xFEC5B4B4, 0xC6D4DB18,
		0xABC5DECD, 0x93D4B161, 0xDBE70195, 0xE3F66E39,
		0x4B80607D, 0x73910FD1, 0x3BA2BF25, 0x03B3D089,
		0xE180B48F, 0xD991DB23, 0x91A26BD7, 0xA9B3047B,
		0x01C50A3F, 0x39D46593, 0x71E7D567, 0x49F6BACB,
		0x24E7BF1E, 0x1CF6D0B2, 0x54C56046, 0x6CD40FEA,
		0xC4A201AE, 0xFCB36E02, 0xB480DEF6, 0x8C91B15A,
		0x750A600B, 0x4D1B0FA7, 0x0528BF53, 0x3D39D0FF,
		0x954FDEBB, 0xAD5EB117, 0xE56D01E3, 0xDD7C6E4F,
		0xB06D6B9A, 0x887C0436, 0xC04FB4C2, 0xF85EDB6E,
		0x5028D52A, 0x6839BA86, 0x200A0A72, 0x181B65DE,
		0xFA2801D8, 0xC2396E74, 0x8A0ADE80, 0xB21BB12C,
		0x1A6DBF68, 0x227CD0C4, 0x6A4F6030, 0x525E0F9C,
		0x3F4F0A49, 0x075E65E5, 0x4F6DD511, 0x777CBABD,
		0xDF0AB4F9, 0xE71BDB55, 0xAF286BA1, 0x9739040D,
		0x59F3BFF2, 0x61E2D05E, 0x29D160AA, 0x11C00F06,
		0xB9B60142, 0x81A76EEE, 0xC994DE1A, 0xF185B1B6,
		0x9C94B463, 0xA485DBCF, 0xECB66B3B, 0xD4A70497,
		0x7CD10AD3, 0x44C0657F, 0x0CF3D58B, 0x34E2BA27,
		0xD6D1DE21, 0xEEC0B18D, 0xA6F30179, 0x9EE26ED5,
		0x36946091, 0x0E850F3D, 0x46B6BFC9, 0x7EA7D065,
		0x13B6D5B0, 0x2BA7BA1C, 0x63940AE8, 0x5B856544,
		0xF3F36B00, 0xCBE204AC, 0x83D1B458, 0xBBC0DBF4,
		0x425B0AA5, 0x7A4A6509, 0x3279D5FD, 0x0A68BA51,
		0xA21EB415, 0x9A0FDBB9, 0xD23C6B4D, 0xEA2D04E1,
		0x873C0134, 0xBF2D6E98, 0xF71EDE6C, 0xCF0FB1C0,
		0x6779BF84, 0x5F68D028, 0x175B60DC, 0x2F4A0F70,
		0xCD796B76, 0xF56804DA, 0xBD5BB42E, 0x854ADB82,
		0x2D3CD5C6, 0x152DBA6A, 0x5D1E0A9E, 0x650F6532,
		0x081E60E7, 0x300F0F4B, 0x783CBFBF, 0x402DD013,
		0xE85BDE57, 0xD04AB1FB, 0x9879010F, 0xA0686EA3
	},
	{
		0x00000000, 0xEF306B19, 0xDB8CA0C3, 0x34BCCBDA,
		0xB2F53777, 0x5DC55C6E, 0x697997B4, 0x8649FCAD,
		0x6006181F, 0x8F367306, 0xBB8AB8DC, 0x54BAD3C5,
		0xD2F32F68, 0x3DC34471, 0x097F8FAB, 0xE64FE4B2,
		0xC00C303E, 0x2F3C5B27, 0x1B8090FD, 0xF4B0FBE4,
		0x72F90749, 0x9DC96C50, 0xA975A78A, 0x4645CC93,
		0xA00A2821, 0x4F3A4338, 0x7B8688E2, 0x94B6E3FB,
		0x12FF1F56, 0xFDCF744F, 0xC973BF95, 0x2643D48C,
		0x85F4168D, 0x6AC47D94, 0x5E78B64E, 0xB148DD57,
		0x370121FA, 0xD8314AE3, 0xEC8D8139, 0x03BDEA20,
		0xE5F20E92, 0x0AC2658B, 0x3E7EAE51, 0xD14EC548,
		0x570739E5, 0xB83752FC, 0x8C8B9926, 0x63BBF23F,
		0x45F826B3, 0xAAC84DAA, 0x9E748670, 0x7144ED69,
		0xF70D11C4, 0x183D7ADD, 0x2C81B107, 0xC3B1DA1E,
		0x25FE3EAC, 0xCACE55B5, 0xFE729E6F, 0x1142F576,
		0x970B09DB, 0x783B62C2, 0x4C87A918, 0xA3B7C201,
		0x0E045BEB, 0xE13430F2, 0xD588FB28, 0x3AB89031,
		0xBCF16C9C, 0x53C10785, 0x677DCC5F, 0x884DA746,
		0x6E0243F4, 0x813228ED, 0xB58EE337, 0x5ABE882E,
		0xDCF77483, 0x33C71F9A, 0x077BD440, 0xE84BBF59,
		0xCE086BD5, 0x213800CC, 0x1584CB16, 0xFAB4A00F,
		0x7CFD5CA2, 0x93CD37BB, 0xA771FC61, 0x48419778,
		0xAE0E73CA, 0x413E18D3, 0x7582D309, 0x9AB2B810,
		0x1CFB44BD, 0xF3CB2FA4, 0xC777E47E, 0x28478F67,
		0x8BF04D66, 0x64C0267F, 0x507CEDA5, 0xBF4C86BC,
		0x39057A11, 0xD6351108, 0xE289DAD2, 0x0DB9B1CB,
		0xEBF65579, 0x04C63E60, 0x307AF5BA, 0xDF4A9EA3,
		0x5903620E, 0xB6330917, 0x828FC2CD, 0x6DBFA9D4,
		0x4BFC7D58, 0xA4CC1641, 0x9070DD9B, 0x7F40B682,
		0xF9094A2F, 0x16392136, 0x2285EAEC, 0xCDB581F5,
		0x2BFA6547, 0xC4CA0E5E, 0xF076C584, 0x1F46AE9D,
		0x990F5230, 0x763F3929, 0x4283F2F3, 0xADB399EA,
		0x1C08B7D6, 0xF338DCCF, 0xC7841715, 0x28B47C0C,
		0xAEFD80A1, 0x41CDEBB8, 0x75712062, 0x9A414B7B,
		0x7C0EAFC9, 0x933EC4D0, 0xA7820F0A, 0x48B26413,
		0xCEFB98BE, 0x21CBF3A7, 0x1577387D, 0xFA475364,
		0xDC0487E8, 0x3334ECF1, 0x0788272B, 0xE8B84C32,
		0x6EF1B09F, 0x81C1DB86, 0xB57D105C, 0x5A4D7B45,
		0xBC029FF7, 0x5332F4EE, 0x678E3F34, 0x88BE542D,
		0x0EF7A880, 0xE1C7C399, 0xD57B0843, 0x3A4B635A,
		0x99FCA15B, 0x76CCCA42, 0x42700198, 0xAD406A81,
		0x2B09962C, 0xC439FD35, 0xF08536EF, 0x1FB55DF6,
		0xF9FAB944, 0x16CAD25D, 0x22761987, 0xCD46729E,
		0x4B0F8E33, 0xA43FE52A, 0x90832EF0, 0x7FB345E9,
		0x59F09165, 0xB6C0FA7C, 0x827C31A6, 0x6D4C5ABF,
		0xEB05A612, 0x0435CD0B, 0x308906D1, 0xDFB96DC8,
		0x39F6897A, 0xD6C6E263, 0xE27A29B9, 0x0D4A42A0,
		0x8B03BE0D, 0x6433D514, 0x508F1ECE, 0xBFBF75D7,
		0x120CEC3D, 0xFD3C8724, 0xC9804CFE, 0x26B027E7,
		0xA0F9DB4A, 0x4FC9B053, 0x7B757B89, 0x94451090,
		0x720AF422, 0x9D3A9F3B, 0xA98654E1, 0x46B63FF8,
		0xC0FFC355, 0x2FCFA84C, 0x1B736396, 0xF443088F,
		0xD200DC03, 0x3D30B71A, 0x098C7CC0, 0xE6BC17D9,
		0x60F5EB74, 0x8FC5806D, 0xBB794BB7, 0x544920AE,
		0xB206C41C, 0x5D36AF05, 0x698A64DF, 0x86BA0FC6,
		0x00F3F36B, 0xEFC39872, 0xDB7F53A8, 0x344F38B1,
		0x97F8FAB0, 0x78C891A9, 0x4C745A73, 0xA344316A,
		0x250DCDC7, 0xCA3DA6DE, 0xFE816D04, 0x11B1061D,
		0xF7FEE2AF, 0x18CE89B6, 0x2C72426C, 0xC3422975,
		0x450BD5D8, 0xAA3BBEC1, 0x9E87751B, 0x71B71E02,
		0x57F4CA8E, 0xB8C4A197, 0x8C786A4D, 0x63480154,
		0xE501FDF9, 0x0A3196E0, 0x3E8D5D3A, 0xD1BD3623,
		0x37F2D291, 0xD8C2B988, 0xEC7E7252, 0x034E194B,
		0x8507E5E6, 0x6A378EFF, 0x5E8B4525, 0xB1BB2E3C
	},
	{
		0x00000000, 0x68032CC8, 0xD0065990, 0xB8057558,
		0xA5E0C5D1, 0xCDE3E919, 0x75E69C41, 0x1DE5B089,
		0x4E2DFD53, 0x262ED19B, 0x9E2BA4C3, 0xF628880B,
		0xEBCD3882, 0x83CE144A, 0x3BCB6112, 0x53C84DDA,
		0x9C5BFAA6, 0xF458D66E, 0x4C5DA336, 0x245E8FFE,
		0x39BB3F77, 0x51B813BF, 0xE9BD66E7, 0x81BE4A2F,
		0xD27607F5, 0xBA752B3D, 0x02705E65, 0x6A7372AD,
		0x7796C224, 0x1F95EEEC, 0xA7909BB4, 0xCF93B77C,
		0x3D5B83BD, 0x5558AF75, 0xED5DDA2D, 0x855EF6E5,
		0x98BB466C, 0xF0B86AA4, 0x48BD1FFC, 0x20BE3334,
		0x73767EEE, 0x1B755226, 0xA370277E, 0xCB730BB6,
		0xD696BB3F, 0xBE9597F7, 0x0690E2AF, 0x6E93CE67,
		0xA100791B, 0xC90355D3, 0x7106208B, 0x19050C43,
		0x04E0BCCA, 0x6CE39002, 0xD4E6E55A, 0xBCE5C992,
		0xEF2D8448, 0x872EA880, 0x3F2BDDD8, 0x5728F110,
		0x4ACD4199, 0x22CE6D51, 0x9ACB1809, 0xF2C834C1,
		0x7AB7077A, 0x12B42BB2, 0xAAB15EEA, 0xC2B27222,
		0xDF57C2AB, 0xB754EE63, 0x0F519B3B, 0x6752B7F3,
		0x349AFA29, 0x5C99D6E1, 0xE49CA3B9, 0x8C9F8F71,
		0x917A3FF8, 0xF9791330, 0x417C6668, 0x297F4AA0,
		0xE6ECFDDC, 0x8EEFD114, 0x36EAA44C, 0x5EE98884,
		0x430C380D, 0x2B0F14C5, 0x930A619D, 0xFB094D55,
		0xA8C1008F, 0xC0C22C47, 0x78C7591F, 0x10C475D7,
		0x0D21C55E, 0x6522E996, 0xDD279CCE, 0xB524B006,
		0x47EC84C7, 0x2FEFA80F, 0x97EADD57, 0xFFE9F19F,
		0xE20C4116, 0x8A0F6DDE, 0x320A1886, 0x5A09344E,
		0x09C17994, 0x61C2555C, 0xD9C72004, 0xB1C40CCC,
		0xAC21BC45, 0xC422908D, 0x7C27E5D5, 0x1424C91D,
		0xDBB77E61, 0xB3B452A9, 0x0BB127F1, 0x63B20B39,
		0x7E57BBB0, 0x16549778, 0xAE51E220, 0xC652CEE8,
		0x959A8332, 0xFD99AFFA, 0x459CDAA2, 0x2D9FF66A,
		0x307A46E3, 0x58796A2B, 0xE07C1F73, 0x887F33BB,
		0xF56E0EF4, 0x9D6D223C, 0x25685764, 0x4D6B7BAC,
		0x508ECB25, 0x388DE7ED, 0x808892B5, 0xE88BBE7D,
		0xBB43F3A7, 0xD340DF6F, 0x6B45AA37, 0x034686FF,
		0x1EA33676, 0x76A01ABE, 0xCEA56FE6, 0xA6A6432E,
		0x6935F452, 0x0136D89A, 0xB933ADC2, 0xD130810A,
		0xCCD53183, 0xA4D61D4B, 0x1CD36813, 0x74D044DB,
		0x27180901, 0x4F1B25C9, 0xF71E5091, 0x9F1D7C59,
		0x82F8CCD0, 0xEAFBE018, 0x52FE9540, 0x3AFDB988,
		0xC8358D49, 0xA036A181, 0x1833D4D9, 0x7030F811,
		0x6DD54898, 0x05D66450, 0xBDD31108, 0xD5D03DC0,
		0x8618701A, 0xEE1B5CD2, 0x561E298A, 0x3E1D0542,
		0x23F8B5CB, 0x4BFB9903, 0xF3FEEC5B, 0x9BFDC093,
		0x546E77EF, 0x3C6D5B27, 0x84682E7F, 0xEC6B02B7,
		0xF18EB23E, 0x998D9EF6, 0x2188EBAE, 0x498BC766,
		0x1A438ABC, 0x7240A674, 0xCA45D32C, 0xA246FFE4,
		0xBFA34F6D, 0xD7A063A5, 0x6FA516FD, 0x07A63A35,
		0x8FD9098E, 0xE7DA2546, 0x5FDF501E, 0x37DC7CD6,
		0x2A39CC5F, 0x423AE097, 0xFA3F95CF, 0x923CB907,
		0xC1F4F4DD, 0xA9F7D815, 0x11F2AD4D, 0x79F18185,
		0x6414310C, 0x0C171DC4, 0xB412689C, 0xDC114454,
		0x1382F328, 0x7B81DFE0, 0xC384AAB8, 0xAB878670,
		0xB66236F9, 0xDE611A31, 0x66646F69, 0x0E6743A1,
		0x5DAF0E7B, 0x35AC22B3, 0x8DA957EB, 0xE5AA7B23,
		0xF84FCBAA, 0x904CE762, 0x2849923A, 0x404ABEF2,
		0xB2828A33, 0xDA81A6FB, 0x6284D3A3, 0x0A87FF6B,
		0x17624FE2, 0x7F61632A, 0xC7641672, 0xAF673ABA,
		0xFCAF7760, 0x94AC5BA8, 0x2CA92EF0, 0x44AA0238,
		0x594FB2B1, 0x314C9E79, 0x8949EB21, 0xE14AC7E9,
		0x2ED97095, 0x46DA5C5D, 0xFEDF2905, 0x96DC05CD,
		0x8B39B544, 0xE33A998C, 0x5B3FECD4, 0x333CC01C,
		0x60F48DC6, 0x08F7A10E, 0xB0F2D456, 0xD8F1F89E,
		0xC5144817, 0xAD1764DF, 0x15121187, 0x7D113D4F
	},
	{
		0x00000000, 0x493C7D27, 0x9278FA4E, 0xDB448769,
		0x211D826D, 0x6821FF4A, 0xB3657823, 0xFA590504,
		0x423B04DA, 0x0B0779FD, 0xD043FE94, 0x997F83B3,
		0x632686B7, 0x2A1AFB90, 0xF15E7CF9, 0xB86201DE,
		0x847609B4, 0xCD4A7493, 0x160EF3FA, 0x5F328EDD,
		0xA56B8BD9, 0xEC57F6FE, 0x37137197, 0x7E2F0CB0,
		0xC64D0D6E, 0x8F717049, 0x5435F720, 0x1D098A07,
		0xE7508F03, 0xAE6CF224, 0x7528754D, 0x3C14086A,
		0x0D006599, 0x443C18BE, 0x9F789FD7, 0xD644E2F0,
		0x2C1DE7F4, 0x65219AD3, 0xBE651DBA, 0xF759609D,
		0x4F3B6143, 0x06071C64, 0xDD439B0D, 0x947FE62A,
		0x6E26E32E, 0x271A9E09, 0xFC5E1960, 0xB5626447,
		0x89766C2D, 0xC04A110A, 0x1B0E9663, 0x5232EB44,
		0xA86BEE40, 0xE1579367, 0x3A13140E, 0x732F6929,
		0xCB4D68F7, 0x827115D0, 0x593592B9, 0x1009EF9E,
		0xEA50EA9A, 0xA36C97BD, 0x782810D4, 0x31146DF3,
		0x1A00CB32, 0x533CB615, 0x8878317C, 0xC1444C5B,
		0x3B1D495F, 0x72213478, 0xA965B311, 0xE059CE36,
		0x583BCFE8, 0x1107B2CF, 0xCA4335A6, 0x837F4881,
		0x79264D85, 0x301A30A2, 0xEB5EB7CB, 0xA262CAEC,
		0x9E76C286, 0xD74ABFA1, 0x0C0E38C8, 0x453245EF,
		0xBF6B40EB, 0xF6573DCC, 0x2D13BAA5, 0x642FC782,
		0xDC4DC65C, 0x9571BB7B, 0x4E353C12, 0x07094135,
		0xFD504431, 0xB46C3916, 0x6F28BE7F, 0x2614C358,
		0x1700AEAB, 0x5E3CD38C, 0x857854E5, 0xCC4429C2,
		0x361D2CC6, 0x7F2151E1, 0xA465D688, 0xED59ABAF,
		0x553BAA71, 0x1C07D756, 0xC743503F, 0x8E7F2D18,
		0x7426281C, 0x3D1A553B, 0xE65ED252, 0xAF62AF75,
		0x9376A71F, 0xDA4ADA38, 0x010E5D51, 0x48322076,
		0xB26B2572, 0xFB575855, 0x2013DF3C, 0x692FA21B,
		0xD14DA3C5, 0x9871DEE2, 0x4335598B, 0x0A0924AC,
		0xF05021A8, 0xB96C5C8F, 0x6228DBE6, 0x2B14A6C1,
		0x34019664, 0x7D3DEB43, 0xA6796C2A, 0xEF45110D,
		0x151C1409, 0x5C20692E, 0x8764EE47, 0xCE589360,
		0x763A92BE, 0x3F06EF99, 0xE44268F0, 0xAD7E15D7,
		0x572710D3, 0x1E1B6DF4, 0xC55FEA9D, 0x8C6397BA,
		0xB0779FD0, 0xF94BE2F7, 0x220F659E, 0x6B3318B9,
		0x916A1DBD, 0xD856609A, 0x0312E7F3, 0x4A2E9AD4,
		0xF24C9B0A, 0xBB70E62D, 0x60346144, 0x29081C63,
		0xD3511967, 0x9A6D6440, 0x4129E329, 0x08159E0E,
		0x3901F3FD, 0x703D8EDA, 0xAB7909B3, 0xE2457494,
		0x181C7190, 0x51200CB7, 0x8A648BDE, 0xC358F6F9,
		0x7B3AF727, 0x32068A00, 0xE9420D69, 0xA07E704E,
		0x5A27754A, 0x131B086D, 0xC85F8F04, 0x8163F223,
		0xBD77FA49, 0xF44B876E, 0x2F0F0007, 0x66337D20,
		0x9C6A7824, 0xD5560503, 0x0E12826A, 0x472EFF4D,
		0xFF4CFE93, 0xB67083B4, 0x6D3404DD, 0x240879FA,
		0xDE517CFE, 0x976D01D9, 0x4C2986B0, 0x0515FB97,
		0x2E015D56, 0x673D2071, 0xBC79A718, 0xF545DA3F,
		0x0F1CDF3B, 0x4620A21C, 0x9D642575, 0xD4585852,
		0x6C3A598C, 0x250624AB, 0xFE42A3C2, 0xB77EDEE5,
		0x4D27DBE1, 0x041BA6C6, 0xDF5F21AF, 0x96635C88,
		0xAA7754E2, 0xE34B29C5, 0x380FAEAC, 0x7133D38B,
		0x8B6AD68F, 0xC256ABA8, 0x19122CC1, 0x502E51E6,
		0xE84C5038, 0xA1702D1F, 0x7A34AA76, 0x3308D751,
		0xC951D255, 0x806DAF72, 0x5B29281B, 0x1215553C,
		0x230138CF, 0x6A3D45E8, 0xB179C281, 0xF845BFA6,
		0x021CBAA2, 0x4B20C785, 0x906440EC, 0xD9583DCB,
		0x613A3C15, 0x28064132, 0xF342C65B, 0xBA7EBB7C,
		0x4027BE78, 0x091BC35F, 0xD25F4436, 0x9B633911,
		0xA777317B, 0xEE4B4C5C, 0x350FCB35, 0x7C33B612,
		0x866AB316, 0xCF56CE31, 0x14124958, 0x5D2E347F,
		0xE54C35A1, 0xAC704886, 0x7734CFEF, 0x3E08B2C8,
		0xC451B7CC, 0x8D6DCAEB, 0x56294D82, 0x1F1530A5
	}
#else							/* !WORDS_BIGENDIAN */
	{
		0x00000000, 0x03836BF2, 0xF7703BE1, 0xF4F35013,
		0x1F979AC7, 0x1C14F135, 0xE8E7A126, 0xEB64CAD4,
		0xCF58D98A, 0xCCDBB278, 0x3828E26B, 0x3BAB8999,
		0xD0CF434D, 0xD34C28BF, 0x27BF78AC, 0x243C135E,
		0x6FC75E10, 0x6C4435E2, 0x98B765F1, 0x9B340E03,
		0x7050C4D7, 0x73D3AF25, 0x8720FF36, 0x84A394C4,
		0xA09F879A, 0xA31CEC68, 0x57EFBC7B, 0x546CD789,
		0xBF081D5D, 0xBC8B76AF, 0x487826BC, 0x4BFB4D4E,
		0xDE8EBD20, 0xDD0DD6D2, 0x29FE86C1, 0x2A7DED33,
		0xC11927E7, 0xC29A4C15, 0x36691C06, 0x35EA77F4,
		0x11D664AA, 0x12550F58, 0xE6A65F4B, 0xE52534B9,
		0x0E41FE6D, 0x0DC2959F, 0xF931C58C, 0xFAB2AE7E,
		0xB149E330, 0xB2CA88C2, 0x4639D8D1, 0x45BAB323,
		0xAEDE79F7, 0xAD5D1205, 0x59AE4216, 0x5A2D29E4,
		0x7E113ABA, 0x7D925148, 0x8961015B, 0x8AE26AA9,
		0x6186A07D, 0x6205CB8F, 0x96F69B9C, 0x9575F06E,
		0xBC1D7B41, 0xBF9E10B3, 0x4B6D40A0, 0x48EE2B52,
		0xA38AE186, 0xA0098A74, 0x54FADA67, 0x5779B195,
		0x7345A2CB, 0x70C6C939, 0x8435992A, 0x87B6F2D8,
		0x6CD2380C, 0x6F5153FE, 0x9BA203ED, 0x9821681F,
		0xD3DA2551, 0xD0594EA3, 0x24AA1EB0, 0x27297542,
		0xCC4DBF96, 0xCFCED464, 0x3B3D8477, 0x38BEEF85,
		0x1C82FCDB, 0x1F019729, 0xEBF2C73A, 0xE871ACC8,
		0x0315661C, 0x00960DEE, 0xF4655DFD, 0xF7E6360F,
		0x6293C661, 0x6110AD93, 0x95E3FD80, 0x96609672,
		0x7D045CA6, 0x7E873754, 0x8A746747, 0x89F70CB5,
		0xADCB1FEB, 0xAE487419, 0x5ABB240A, 0x59384FF8,
		0xB25C852C, 0xB1DFEEDE, 0x452CBECD, 0x46AFD53F,
		0x0D549871, 0x0ED7F383, 0xFA24A390, 0xF9A7C862,
		0x12C302B6, 0x11406944, 0xE5B33957, 0xE63052A5,
		0xC20C41FB, 0xC18F2A09, 0x357C7A1A, 0x36FF11E8,
		0xDD9BDB3C, 0xDE18B0CE, 0x2AEBE0DD, 0x29688B2F,
		0x783BF682, 0x7BB89D70, 0x8F4BCD63, 0x8CC8A691,
		0x67AC6C45, 0x642F07B7, 0x90DC57A4, 0x935F3C56,
		0xB7632F08, 0xB4E044FA, 0x401314E9, 0x43907F1B,
		0xA8F4B5CF, 0xAB77DE3D, 0x5F848E2E, 0x5C07E5DC,
		0x17FCA892, 0x147FC360, 0xE08C9373, 0xE30FF881,
		0x086B3255, 0x0BE859A7, 0xFF1B09B4, 0xFC986246,
		0xD8A47118, 0xDB271AEA, 0x2FD44AF9, 0x2C57210B,
		0xC733EBDF, 0xC4B0802D, 0x3043D03E, 0x33C0BBCC,
		0xA6B54BA2, 0xA5362050, 0x51C57043, 0x52461BB1,
		0xB922D165, 0xBAA1BA97, 0x4E52EA84, 0x4DD18176,
		0x69ED9228, 0x6A6EF9DA, 0x9E9DA9C9, 0x9D1EC23B,
		0x767A08EF, 0x75F9631D, 0x810A330E, 0x828958FC,
		0xC97215B2, 0xCAF17E40, 0x3E022E53, 0x3D8145A1,
		0xD6E58F75, 0xD566E487, 0x2195B494, 0x2216DF66,
		0x062ACC38, 0x05A9A7CA, 0xF15AF7D9, 0xF2D99C2B,
		0x19BD56FF, 0x1A3E3D0D, 0xEECD6D1E, 0xED4E06EC,
		0xC4268DC3, 0xC7A5E631, 0x3356B622, 0x30D5DDD0,
		0xDBB11704, 0xD8327CF6, 0x2CC12CE5, 0x2F424717,
		0x0B7E5449, 0x08FD3FBB, 0xFC0E6FA8, 0xFF8D045A,
		0x14E9CE8E, 0x176AA57C, 0xE399F56F, 0xE01A9E9D,
		0xABE1D3D3, 0xA862B821, 0x5C91E832, 0x5F1283C0,
		0xB4764914, 0xB7F522E6, 0x430672F5, 0x40851907,
		0x64B90A59, 0x673A61AB, 0x93C931B8, 0x904A5A4A,
		0x7B2E909E, 0x78ADFB6C, 0x8C5EAB7F, 0x8FDDC08D,
		0x1AA830E3, 0x192B5B11, 0xEDD80B02, 0xEE5B60F0,
		0x053FAA24, 0x06BCC1D6, 0xF24F91C5, 0xF1CCFA37,
		0xD5F0E969, 0xD673829B, 0x2280D288, 0x2103B97A,
		0xCA6773AE, 0xC9E4185C, 0x3D17484F, 0x3E9423BD,
		0x756F6EF3, 0x76EC0501, 0x821F5512, 0x819C3EE0,
		0x6AF8F434, 0x697B9FC6, 0x9D88CFD5, 0x9E0BA427,
		0xBA37B779, 0xB9B4DC8B, 0x4D478C98, 0x4EC4E76A,
		0xA5A02DBE, 0xA623464C, 0x52D0165F, 0x51537DAD,
	},
	{
		0x00000000, 0x7798A213, 0xEE304527, 0x99A8E734,
		0xDC618A4E, 0xABF9285D, 0x3251CF69, 0x45C96D7A,
		0xB8C3149D, 0xCF5BB68E, 0x56F351BA, 0x216BF3A9,
		0x64A29ED3, 0x133A3CC0, 0x8A92DBF4, 0xFD0A79E7,
		0x81F1C53F, 0xF669672C, 0x6FC18018, 0x1859220B,
		0x5D904F71, 0x2A08ED62, 0xB3A00A56, 0xC438A845,
		0x3932D1A2, 0x4EAA73B1, 0xD7029485, 0xA09A3696,
		0xE5535BEC, 0x92CBF9FF, 0x0B631ECB, 0x7CFBBCD8,
		0x02E38B7F, 0x757B296C, 0xECD3CE58, 0x9B4B6C4B,
		0xDE820131, 0xA91AA322, 0x30B24416, 0x472AE605,
		0xBA209FE2, 0xCDB83DF1, 0x5410DAC5, 0x238878D6,
		0x664115AC, 0x11D9B7BF, 0x8871508B, 0xFFE9F298,
		0x83124E40, 0xF48AEC53, 0x6D220B67, 0x1ABAA974,
		0x5F73C40E, 0x28EB661D, 0xB1438129, 0xC6DB233A,
		0x3BD15ADD, 0x4C49F8CE, 0xD5E11FFA, 0xA279BDE9,
		0xE7B0D093, 0x90287280, 0x098095B4, 0x7E1837A7,
		0x04C617FF, 0x735EB5EC, 0xEAF652D8, 0x9D6EF0CB,
		0xD8A79DB1, 0xAF3F3FA2, 0x3697D896, 0x410F7A85,
		0xBC050362, 0xCB9DA171, 0x52354645, 0x25ADE456,
		0x6064892C, 0x17FC2B3F, 0x8E54CC0B, 0xF9CC6E18,
		0x8537D2C0, 0xF2AF70D3, 0x6B0797E7, 0x1C9F35F4,
		0x5956588E, 0x2ECEFA9D, 0xB7661DA9, 0xC0FEBFBA,
		0x3DF4C65D, 0x4A6C644E, 0xD3C4837A, 0xA45C2169,
		0xE1954C13, 0x960DEE00, 0x0FA50934, 0x783DAB27,
		0x06259C80, 0x71BD3E93, 0xE815D9A7, 0x9F8D7BB4,
		0xDA4416CE, 0xADDCB4DD, 0x347453E9, 0x43ECF1FA,
		0xBEE6881D, 0xC97E2A0E, 0x50D6CD3A, 0x274E6F29,
		0x62870253, 0x151FA040, 0x8CB74774, 0xFB2FE567,
		0x87D459BF, 0xF04CFBAC, 0x69E41C98, 0x1E7CBE8B,
		0x5BB5D3F1, 0x2C2D71E2, 0xB58596D6, 0xC21D34C5,
		0x3F174D22, 0x488FEF31, 0xD1270805, 0xA6BFAA16,
		0xE376C76C, 0x94EE657F, 0x0D46824B, 0x7ADE2058,
		0xF9FAC3FB, 0x8E6261E8, 0x17CA86DC, 0x605224CF,
		0x259B49B5, 0x5203EBA6, 0xCBAB0C92, 0xBC33AE81,
		0x4139D766, 0x36A17575, 0xAF099241, 0xD8913052,
		0x9D585D28, 0xEAC0FF3B, 0x7368180F, 0x04F0BA1C,
		0x780B06C4, 0x0F93A4D7, 0x963B43E3, 0xE1A3E1F0,
		0xA46A8C8A, 0xD3F22E99, 0x4A5AC9AD, 0x3DC26BBE,
		0xC0C81259, 0xB750B04A, 0x2EF8577E, 0x5960F56D,
		0x1CA99817, 0x6B313A04, 0xF299DD30, 0x85017F23,
		0xFB194884, 0x8C81EA97, 0x15290DA3, 0x62B1AFB0,
		0x2778C2CA, 0x50E060D9, 0xC94887ED, 0xBED025FE,
		0x43DA5C19, 0x3442FE0A, 0xADEA193E, 0xDA72BB2D,
		0x9FBBD657, 0xE8237444, 0x718B9370, 0x06133163,
		0x7AE88DBB, 0x0D702FA8, 0x94D8C89C, 0xE3406A8F,
		0xA68907F5, 0xD111A5E6, 0x48B942D2, 0x3F21E0C1,
		0xC22B9926, 0xB5B33B35, 0x2C1BDC01, 0x5B837E12,
		0x1E4A1368, 0x69D2B17B, 0xF07A564F, 0x87E2F45C,
		0xFD3CD404, 0x8AA47617, 0x130C9123, 0x64943330,
		0x215D5E4A, 0x56C5FC59, 0xCF6D1B6D, 0xB8F5B97E,
		0x45FFC099, 0x3267628A, 0xABCF85BE, 0xDC5727AD,
		0x999E4AD7, 0xEE06E8C4, 0x77AE0FF0, 0x0036ADE3,
		0x7CCD113B, 0x0B55B328, 0x92FD541C, 0xE565F60F,
		0xA0AC9B75, 0xD7343966, 0x4E9CDE52, 0x39047C41,
		0xC40E05A6, 0xB396A7B5, 0x2A3E4081, 0x5DA6E292,
		0x186F8FE8, 0x6FF72DFB, 0xF65FCACF, 0x81C768DC,
		0xFFDF5F7B, 0x8847FD68, 0x11EF1A5C, 0x6677B84F,
		0x23BED535, 0x54267726, 0xCD8E9012, 0xBA163201,
		0x471C4BE6, 0x3084E9F5, 0xA92C0EC1, 0xDEB4ACD2,
		0x9B7DC1A8, 0xECE563BB, 0x754D848F, 0x02D5269C,
		0x7E2E9A44, 0x09B63857, 0x901EDF63, 0xE7867D70,
		0xA24F100A, 0xD5D7B219, 0x4C7F552D, 0x3BE7F73E,
		0xC6ED8ED9, 0xB1752CCA, 0x28DDCBFE, 0x5F4569ED,
		0x1A8C0497, 0x6D14A684, 0xF4BC41B0, 0x8324E3A3,
	},
	{
		0x00000000, 0x7E9241A5, 0x0D526F4F, 0x73C02EEA,
		0x1AA4DE9E, 0x64369F3B, 0x17F6B1D1, 0x6964F074,
		0xC53E5138, 0xBBAC109D, 0xC86C3E77, 0xB6FE7FD2,
		0xDF9A8FA6, 0xA108CE03, 0xD2C8E0E9, 0xAC5AA14C,
		0x8A7DA270, 0xF4EFE3D5, 0x872FCD3F, 0xF9BD8C9A,
		0x90D97CEE, 0xEE4B3D4B, 0x9D8B13A1, 0xE3195204,
		0x4F43F348, 0x31D1B2ED, 0x42119C07, 0x3C83DDA2,
		0x55E72DD6, 0x2B756C73, 0x58B54299, 0x2627033C,
		0x14FB44E1, 0x6A690544, 0x19A92BAE, 0x673B6A0B,
		0x0E5F9A7F, 0x70CDDBDA, 0x030DF530, 0x7D9FB495,
		0xD1C515D9, 0xAF57547C, 0xDC977A96, 0xA2053B33,
		0xCB61CB47, 0xB5F38AE2, 0xC633A408, 0xB8A1E5AD,
		0x9E86E691, 0xE014A734, 0x93D489DE, 0xED46C87B,
		0x8422380F, 0xFAB079AA, 0x89705740, 0xF7E216E5,
		0x5BB8B7A9, 0x252AF60C, 0x56EAD8E6, 0x28789943,
		0x411C6937, 0x3F8E2892, 0x4C4E0678, 0x32DC47DD,
		0xD98065C7, 0xA7122462, 0xD4D20A88, 0xAA404B2D,
		0xC324BB59, 0xBDB6FAFC, 0xCE76D416, 0xB0E495B3,
		0x1CBE34FF, 0x622C755A, 0x11EC5BB0, 0x6F7E1A15,
		0x061AEA61, 0x7888ABC4, 0x0B48852E, 0x75DAC48B,
		0x53FDC7B7, 0x2D6F8612, 0x5EAFA8F8, 0x203DE95D,
		0x49591929, 0x37CB588C, 0x440B7666, 0x3A9937C3,
		0x96C3968F, 0xE851D72A, 0x9B91F9C0, 0xE503B865,
		0x8C674811, 0xF2F509B4, 0x8135275E, 0xFFA766FB,
		0xCD7B2126, 0xB3E96083, 0xC0294E69, 0xBEBB0FCC,
		0xD7DFFFB8, 0xA94DBE1D, 0xDA8D90F7, 0xA41FD152,
		0x0845701E, 0x76D731BB, 0x05171F51, 0x7B855EF4,
		0x12E1AE80, 0x6C73EF25, 0x1FB3C1CF, 0x6121806A,
		0x47068356, 0x3994C2F3, 0x4A54EC19, 0x34C6ADBC,
		0x5DA25DC8, 0x23301C6D, 0x50F03287, 0x2E627322,
		0x8238D26E, 0xFCAA93CB, 0x8F6ABD21, 0xF1F8FC84,
		0x989C0CF0, 0xE60E4D55, 0x95CE63BF, 0xEB5C221A,
		0x4377278B, 0x3DE5662E, 0x4E2548C4, 0x30B70961,
		0x59D3F915, 0x2741B8B0, 0x5481965A, 0x2A13D7FF,
		0x864976B3, 0xF8DB3716, 0x8B1B19FC, 0xF5895859,
		0x9CEDA82D, 0xE27FE988, 0x91BFC762, 0xEF2D86C7,
		0xC90A85FB, 0xB798C45E, 0xC458EAB4, 0xBACAAB11,
		0xD3AE5B65, 0xAD3C1AC0, 0xDEFC342A, 0xA06E758F,
		0x0C34D4C3, 0x72A69566, 0x0166BB8C, 0x7FF4FA29,
		0x16900A5D, 0x68024BF8, 0x1BC26512, 0x655024B7,
		0x578C636A, 0x291E22CF, 0x5ADE0C25, 0x244C4D80,
		0x4D28BDF4, 0x33BAFC51, 0x407AD2BB, 0x3EE8931E,
		0x92B23252, 0xEC2073F7, 0x9FE05D1D, 0xE1721CB8,
		0x8816ECCC, 0xF684AD69, 0x85448383, 0xFBD6C226,
		0xDDF1C11A, 0xA36380BF, 0xD0A3AE55, 0xAE31EFF0,
		0xC7551F84, 0xB9C75E21, 0xCA0770CB, 0xB495316E,
		0x18CF9022, 0x665DD187, 0x159DFF6D, 0x6B0FBEC8,
		0x026B4EBC, 0x7CF90F19, 0x0F3921F3, 0x71AB6056,
		0x9AF7424C, 0xE46503E9, 0x97A52D03, 0xE9376CA6,
		0x80539CD2, 0xFEC1DD77, 0x8D01F39D, 0xF393B238,
		0x5FC91374, 0x215B52D1, 0x529B7C3B, 0x2C093D9E,
		0x456DCDEA, 0x3BFF8C4F, 0x483FA2A5, 0x36ADE300,
		0x108AE03C, 0x6E18A199, 0x1DD88F73, 0x634ACED6,
		0x0A2E3EA2, 0x74BC7F07, 0x077C51ED, 0x79EE1048,
		0xD5B4B104, 0xAB26F0A1, 0xD8E6DE4B, 0xA6749FEE,
		0xCF106F9A, 0xB1822E3F, 0xC24200D5, 0xBCD04170,
		0x8E0C06AD, 0xF09E4708, 0x835E69E2, 0xFDCC2847,
		0x94A8D833, 0xEA3A9996, 0x99FAB77C, 0xE768F6D9,
		0x4B325795, 0x35A01630, 0x466038DA, 0x38F2797F,
		0x5196890B, 0x2F04C8AE, 0x5CC4E644, 0x2256A7E1,
		0x0471A4DD, 0x7AE3E578, 0x0923CB92, 0x77B18A37,
		0x1ED57A43, 0x60473BE6, 0x1387150C, 0x6D1554A9,
		0xC14FF5E5, 0xBFDDB440, 0xCC1D9AAA, 0xB28FDB0F,
		0xDBEB2B7B, 0xA5796ADE, 0xD6B94434, 0xA82B0591,
	},
	{
		0x00000000, 0xB8AA45DD, 0x812367BF, 0x39892262,
		0xF331227B, 0x4B9B67A6, 0x721245C4, 0xCAB80019,
		0xE66344F6, 0x5EC9012B, 0x67402349, 0xDFEA6694,
		0x1552668D, 0xADF82350, 0x94710132, 0x2CDB44EF,
		0x3DB164E9, 0x851B2134, 0xBC920356, 0x0438468B,
		0xCE804692, 0x762A034F, 0x4FA3212D, 0xF70964F0,
		0xDBD2201F, 0x637865C2, 0x5AF147A0, 0xE25B027D,
		0x28E30264, 0x904947B9, 0xA9C065DB, 0x116A2006,
		0x8B1425D7, 0x33BE600A, 0x0A374268, 0xB29D07B5,
		0x782507AC, 0xC08F4271, 0xF9066013, 0x41AC25CE,
		0x6D776121, 0xD5DD24FC, 0xEC54069E, 0x54FE4343,
		0x9E46435A, 0x26EC0687, 0x1F6524E5, 0xA7CF6138,
		0xB6A5413E, 0x0E0F04E3, 0x37862681, 0x8F2C635C,
		0x45946345, 0xFD3E2698, 0xC4B704FA, 0x7C1D4127,
		0x50C605C8, 0xE86C4015, 0xD1E56277, 0x694F27AA,
		0xA3F727B3, 0x1B5D626E, 0x22D4400C, 0x9A7E05D1,
		0xE75FA6AB, 0x5FF5E376, 0x667CC114, 0xDED684C9,
		0x146E84D0, 0xACC4C10D, 0x954DE36F, 0x2DE7A6B2,
		0x013CE25D, 0xB996A780, 0x801F85E2, 0x38B5C03F,
		0xF20DC026, 0x4AA785FB, 0x732EA799, 0xCB84E244,
		0xDAEEC242, 0x6244879F, 0x5BCDA5FD, 0xE367E020,
		0x29DFE039, 0x9175A5E4, 0xA8FC8786, 0x1056C25B,
		0x3C8D86B4, 0x8427C369, 0xBDAEE10B, 0x0504A4D6,
		0xCFBCA4CF, 0x7716E112, 0x4E9FC370, 0xF63586AD,
		0x6C4B837C, 0xD4E1C6A1, 0xED68E4C3, 0x55C2A11E,
		0x9F7AA107, 0x27D0E4DA, 0x1E59C6B8, 0xA6F38365,
		0x8A28C78A, 0x32828257, 0x0B0BA035, 0xB3A1E5E8,
		0x7919E5F1, 0xC1B3A02C, 0xF83A824E, 0x4090C793,
		0x51FAE795, 0xE950A248, 0xD0D9802A, 0x6873C5F7,
		0xA2CBC5EE, 0x1A618033, 0x23E8A251, 0x9B42E78C,
		0xB799A363, 0x0F33E6BE, 0x36BAC4DC, 0x8E108101,
		0x44A88118, 0xFC02C4C5, 0xC58BE6A7, 0x7D21A37A,
		0x3FC9A052, 0x8763E58F, 0xBEEAC7ED, 0x06408230,
		0xCCF88229, 0x7452C7F4, 0x4DDBE596, 0xF571A04B,
		0xD9AAE4A4, 0x6100A179, 0x5889831B, 0xE023C6C6,
		0x2A9BC6DF, 0x92318302, 0xABB8A160, 0x1312E4BD,
		0x0278C4BB, 0xBAD28166, 0x835BA304, 0x3BF1E6D9,
		0xF149E6C0, 0x49E3A31D, 0x706A817F, 0xC8C0C4A2,
		0xE41B804D, 0x5CB1C590, 0x6538E7F2, 0xDD92A22F,
		0x172AA236, 0xAF80E7EB, 0x9609C589, 0x2EA38054,
		0xB4DD8585, 0x0C77C058, 0x35FEE23A, 0x8D54A7E7,
		0x47ECA7FE, 0xFF46E223, 0xC6CFC041, 0x7E65859C,
		0x52BEC173, 0xEA1484AE, 0xD39DA6CC, 0x6B37E311,
		0xA18FE308, 0x1925A6D5, 0x20AC84B7, 0x9806C16A,
		0x896CE16C, 0x31C6A4B1, 0x084F86D3, 0xB0E5C30E,
		0x7A5DC317, 0xC2F786CA, 0xFB7EA4A8, 0x43D4E175,
		0x6F0FA59A, 0xD7A5E047, 0xEE2CC225, 0x568687F8,
		0x9C3E87E1, 0x2494C23C, 0x1D1DE05E, 0xA5B7A583,
		0xD89606F9, 0x603C4324, 0x59B56146, 0xE11F249B,
		0x2BA72482, 0x930D615F, 0xAA84433D, 0x122E06E0,
		0x3EF5420F, 0x865F07D2, 0xBFD625B0, 0x077C606D,
		0xCDC46074, 0x756E25A9, 0x4CE707CB, 0xF44D4216,
		0xE5276210, 0x5D8D27CD, 0x640405AF, 0xDCAE4072,
		0x1616406B, 0xAEBC05B6, 0x973527D4, 0x2F9F6209,
		0x034426E6, 0xBBEE633B, 0x82674159, 0x3ACD0484,
		0xF075049D, 0x48DF4140, 0x71566322, 0xC9FC26FF,
		0x5382232E, 0xEB2866F3, 0xD2A14491, 0x6A0B014C,
		0xA0B30155, 0x18194488, 0x219066EA, 0x993A2337,
		0xB5E167D8, 0x0D4B2205, 0x34C20067, 0x8C6845BA,
		0x46D045A3, 0xFE7A007E, 0xC7F3221C, 0x7F5967C1,
		0x6E3347C7, 0xD699021A, 0xEF102078, 0x57BA65A5,
		0x9D0265BC, 0x25A82061, 0x1C210203, 0xA48B47DE,
		0x88500331, 0x30FA46EC, 0x0973648E, 0xB1D92153,
		0x7B61214A, 0xC3CB6497, 0xFA4246F5, 0x42E80328,
	},
	{
		0x00000000, 0xAC6F1138, 0x58DF2270, 0xF4B03348,
		0xB0BE45E0, 0x1CD154D8, 0xE8616790, 0x440E76A8,
		0x910B67C5, 0x3D6476FD, 0xC9D445B5, 0x65BB548D,
		0x21B52225, 0x8DDA331D, 0x796A0055, 0xD505116D,
		0xD361228F, 0x7F0E33B7, 0x8BBE00FF, 0x27D111C7,
		0x63DF676F, 0xCFB07657, 0x3B00451F, 0x976F5427,
		0x426A454A, 0xEE055472, 0x1AB5673A, 0xB6DA7602,
		0xF2D400AA, 0x5EBB1192, 0xAA0B22DA, 0x066433E2,
		0x57B5A81B, 0xFBDAB923, 0x0F6A8A6B, 0xA3059B53,
		0xE70BEDFB, 0x4B64FCC3, 0xBFD4CF8B, 0x13BBDEB3,
		0xC6BECFDE, 0x6AD1DEE6, 0x9E61EDAE, 0x320EFC96,
		0x76008A3E, 0xDA6F9B06, 0x2EDFA84E, 0x82B0B976,
		0x84D48A94, 0x28BB9BAC, 0xDC0BA8E4, 0x7064B9DC,
		0x346ACF74, 0x9805DE4C, 0x6CB5ED04, 0xC0DAFC3C,
		0x15DFED51, 0xB9B0FC69, 0x4D00CF21, 0xE16FDE19,
		0xA561A8B1, 0x090EB989, 0xFDBE8AC1, 0x51D19BF9,
		0xAE6A5137, 0x0205400F, 0xF6B57347, 0x5ADA627F,
		0x1ED414D7, 0xB2BB05EF, 0x460B36A7, 0xEA64279F,
		0x3F6136F2, 0x930E27CA, 0x67BE1482, 0xCBD105BA,
		0x8FDF7312, 0x23B0622A, 0xD7005162, 0x7B6F405A,
		0x7D0B73B8, 0xD1646280, 0x25D451C8, 0x89BB40F0,
		0xCDB53658, 0x61DA2760, 0x956A1428, 0x39050510,
		0xEC00147D, 0x406F0545, 0xB4DF360D, 0x18B02735,
		0x5CBE519D, 0xF0D140A5, 0x046173ED, 0xA80E62D5,
		0xF9DFF92C, 0x55B0E814, 0xA100DB5C, 0x0D6FCA64,
		0x4961BCCC, 0xE50EADF4, 0x11BE9EBC, 0xBDD18F84,
		0x68D49EE9, 0xC4BB8FD1, 0x300BBC99, 0x9C64ADA1,
		0xD86ADB09, 0x7405CA31, 0x80B5F979, 0x2CDAE841,
		0x2ABEDBA3, 0x86D1CA9B, 0x7261F9D3, 0xDE0EE8EB,
		0x9A009E43, 0x366F8F7B, 0xC2DFBC33, 0x6EB0AD0B,
		0xBBB5BC66, 0x17DAAD5E, 0xE36A9E16, 0x4F058F2E,
		0x0B0BF986, 0xA764E8BE, 0x53D4DBF6, 0xFFBBCACE,
		0x5CD5A26E, 0xF0BAB356, 0x040A801E, 0xA8659126,
		0xEC6BE78E, 0x4004F6B6, 0xB4B4C5FE, 0x18DBD4C6,
		0xCDDEC5AB, 0x61B1D493, 0x9501E7DB, 0x396EF6E3,
		0x7D60804B, 0xD10F9173, 0x25BFA23B, 0x89D0B303,
		0x8FB480E1, 0x23DB91D9, 0xD76BA291, 0x7B04B3A9,
		0x3F0AC501, 0x9365D439, 0x67D5E771, 0xCBBAF649,
		0x1EBFE724, 0xB2D0F61C, 0x4660C554, 0xEA0FD46C,
		0xAE01A2C4, 0x026EB3FC, 0xF6DE80B4, 0x5AB1918C,
		0x0B600A75, 0xA70F1B4D, 0x53BF2805, 0xFFD0393D,
		0xBBDE4F95, 0x17B15EAD, 0xE3016DE5, 0x4F6E7CDD,
		0x9A6B6DB0, 0x36047C88, 0xC2B44FC0, 0x6EDB5EF8,
		0x2AD52850, 0x86BA3968, 0x720A0A20, 0xDE651B18,
		0xD80128FA, 0x746E39C2, 0x80DE0A8A, 0x2CB11BB2,
		0x68BF6D1A, 0xC4D07C22, 0x30604F6A, 0x9C0F5E52,
		0x490A4F3F, 0xE5655E07, 0x11D56D4F, 0xBDBA7C77,
		0xF9B40ADF, 0x55DB1BE7, 0xA16B28AF, 0x0D043997,
		0xF2BFF359, 0x5ED0E261, 0xAA60D129, 0x060FC011,
		0x4201B6B9, 0xEE6EA781, 0x1ADE94C9, 0xB6B185F1,
		0x63B4949C, 0xCFDB85A4, 0x3B6BB6EC, 0x9704A7D4,
		0xD30AD17C, 0x7F65C044, 0x8BD5F30C, 0x27BAE234,
		0x21DED1D6, 0x8DB1C0EE, 0x7901F3A6, 0xD56EE29E,
		0x91609436, 0x3D0F850E, 0xC9BFB646, 0x65D0A77E,
		0xB0D5B613, 0x1CBAA72B, 0xE80A9463, 0x4465855B,
		0x006BF3F3, 0xAC04E2CB, 0x58B4D183, 0xF4DBC0BB,
		0xA50A5B42, 0x09654A7A, 0xFDD57932, 0x51BA680A,
		0x15B41EA2, 0xB9DB0F9A, 0x4D6B3CD2, 0xE1042DEA,
		0x34013C87, 0x986E2DBF, 0x6CDE1EF7, 0xC0B10FCF,
		0x84BF7967, 0x28D0685F, 0xDC605B17, 0x700F4A2F,
		0x766B79CD, 0xDA0468F5, 0x2EB45BBD, 0x82DB4A85,
		0xC6D53C2D, 0x6ABA2D15, 0x9E0A1E5D, 0x32650F65,
		0xE7601E08, 0x4B0F0F30, 0xBFBF3C78, 0x13D02D40,
		0x57DE5BE8, 0xFBB14AD0, 0x0F017998, 0xA36E68A0,
	},
	{
		0x00000000, 0x196B30EF, 0xC3A08CDB, 0xDACBBC34,
		0x7737F5B2, 0x6E5CC55D, 0xB4977969, 0xADFC4986,
		0x1F180660, 0x0673368F, 0xDCB88ABB, 0xC5D3BA54,
		0x682FF3D2, 0x7144C33D, 0xAB8F7F09, 0xB2E44FE6,
		0x3E300CC0, 0x275B3C2F, 0xFD90801B, 0xE4FBB0F4,
		0x4907F972, 0x506CC99D, 0x8AA775A9, 0x93CC4546,
		0x21280AA0, 0x38433A4F, 0xE288867B, 0xFBE3B694,
		0x561FFF12, 0x4F74CFFD, 0x95BF73C9, 0x8CD44326,
		0x8D16F485, 0x947DC46A, 0x4EB6785E, 0x57DD48B1,
		0xFA210137, 0xE34A31D8, 0x39818DEC, 0x20EABD03,
		0x920EF2E5, 0x8B65C20A, 0x51AE7E3E, 0x48C54ED1,
		0xE5390757, 0xFC5237B8, 0x26998B8C, 0x3FF2BB63,
		0xB326F845, 0xAA4DC8AA, 0x7086749E, 0x69ED4471,
		0xC4110DF7, 0xDD7A3D18, 0x07B1812C, 0x1EDAB1C3,
		0xAC3EFE25, 0xB555CECA, 0x6F9E72FE, 0x76F54211,
		0xDB090B97, 0xC2623B78, 0x18A9874C, 0x01C2B7A3,
		0xEB5B040E, 0xF23034E1, 0x28FB88D5, 0x3190B83A,
		0x9C6CF1BC, 0x8507C153, 0x5FCC7D67, 0x46A74D88,
		0xF443026E, 0xED283281, 0x37E38EB5, 0x2E88BE5A,
		0x8374F7DC, 0x9A1FC733, 0x40D47B07, 0x59BF4BE8,
		0xD56B08CE, 0xCC003821, 0x16CB8415, 0x0FA0B4FA,
		0xA25CFD7C, 0xBB37CD93, 0x61FC71A7, 0x78974148,
		0xCA730EAE, 0xD3183E41, 0x09D38275, 0x10B8B29A,
		0xBD44FB1C, 0xA42FCBF3, 0x7EE477C7, 0x678F4728,
		0x664DF08B, 0x7F26C064, 0xA5ED7C50, 0xBC864CBF,
		0x117A0539, 0x081135D6, 0xD2DA89E2, 0xCBB1B90D,
		0x7955F6EB, 0x603EC604, 0xBAF57A30, 0xA39E4ADF,
		0x0E620359, 0x170933B6, 0xCDC28F82, 0xD4A9BF6D,
		0x587DFC4B, 0x4116CCA4, 0x9BDD7090, 0x82B6407F,
		0x2F4A09F9, 0x36213916, 0xECEA8522, 0xF581B5CD,
		0x4765FA2B, 0x5E0ECAC4, 0x84C576F0, 0x9DAE461F,
		0x30520F99, 0x29393F76, 0xF3F28342, 0xEA99B3AD,
		0xD6B7081C, 0xCFDC38F3, 0x151784C7, 0x0C7CB428,
		0xA180FDAE, 0xB8EBCD41, 0x62207175, 0x7B4B419A,
		0xC9AF0E7C, 0xD0C43E93, 0x0A0F82A7, 0x1364B248,
		0xBE98FBCE, 0xA7F3CB21, 0x7D387715, 0x645347FA,
		0xE88704DC, 0xF1EC3433, 0x2B278807, 0x324CB8E8,
		0x9FB0F16E, 0x86DBC181, 0x5C107DB5, 0x457B4D5A,
		0xF79F02BC, 0xEEF43253, 0x343F8E67, 0x2D54BE88,
		0x80A8F70E, 0x99C3C7E1, 0x43087BD5, 0x5A634B3A,
		0x5BA1FC99, 0x42CACC76, 0x98017042, 0x816A40AD,
		0x2C96092B, 0x35FD39C4, 0xEF3685F0, 0xF65DB51F,
		0x44B9FAF9, 0x5DD2CA16, 0x87197622, 0x9E7246CD,
		0x338E0F4B, 0x2AE53FA4, 0xF02E8390, 0xE945B37F,
		0x6591F059, 0x7CFAC0B6, 0xA6317C82, 0xBF5A4C6D,
		0x12A605EB, 0x0BCD3504, 0xD1068930, 0xC86DB9DF,
		0x7A89F639, 0x63E2C6D6, 0xB9297AE2, 0xA0424A0D,
		0x0DBE038B, 0x14D53364, 0xCE1E8F50, 0xD775BFBF,
		0x3DEC0C12, 0x24873CFD, 0xFE4C80C9, 0xE727B026,
		0x4ADBF9A0, 0x53B0C94F, 0x897B757B, 0x90104594,
		0x22F40A72, 0x3B9F3A9D, 0xE15486A9, 0xF83FB646,
		0x55C3FFC0, 0x4CA8CF2F, 0x9663731B, 0x8F0843F4,
		0x03DC00D2, 0x1AB7303D, 0xC07C8C09, 0xD917BCE6,
		0x74EBF560, 0x6D80C58F, 0xB74B79BB, 0xAE204954,
		0x1CC406B2, 0x05AF365D, 0xDF648A69, 0xC60FBA86,
		0x6BF3F300, 0x7298C3EF, 0xA8537FDB, 0xB1384F34,
		0xB0FAF897, 0xA991C878, 0x735A744C, 0x6A3144A3,
		0xC7CD0D25, 0xDEA63DCA, 0x046D81FE, 0x1D06B111,
		0xAFE2FEF7, 0xB689CE18, 0x6C42722C, 0x752942C3,
		0xD8D50B45, 0xC1BE3BAA, 0x1B75879E, 0x021EB771,
		0x8ECAF457, 0x97A1C4B8, 0x4D6A788C, 0x54014863,
		0xF9FD01E5, 0xE096310A, 0x3A5D8D3E, 0x2336BDD1,
		0x91D2F237, 0x88B9C2D8, 0x52727EEC, 0x4B194E03,
		0xE6E50785, 0xFF8E376A, 0x25458B5E, 0x3C2EBBB1,
	},
	{
		0x00000000, 0xC82C0368, 0x905906D0, 0x587505B8,
		0xD1C5E0A5, 0x19E9E3CD, 0x419CE675, 0x89B0E51D,
		0x53FD2D4E, 0x9BD12E26, 0xC3A42B9E, 0x0B8828F6,
		0x8238CDEB, 0x4A14CE83, 0x1261CB3B, 0xDA4DC853,
		0xA6FA5B9C, 0x6ED658F4, 0x36A35D4C, 0xFE8F5E24,
		0x773FBB39, 0xBF13B851, 0xE766BDE9, 0x2F4ABE81,
		0xF50776D2, 0x3D2B75BA, 0x655E7002, 0xAD72736A,
		0x24C29677, 0xECEE951F, 0xB49B90A7, 0x7CB793CF,
		0xBD835B3D, 0x75AF5855, 0x2DDA5DED, 0xE5F65E85,
		0x6C46BB98, 0xA46AB8F0, 0xFC1FBD48, 0x3433BE20,
		0xEE7E7673, 0x2652751B, 0x7E2770A3, 0xB60B73CB,
		0x3FBB96D6, 0xF79795BE, 0xAFE29006, 0x67CE936E,
		0x1B7900A1, 0xD35503C9, 0x8B200671, 0x430C0519,
		0xCABCE004, 0x0290E36C, 0x5AE5E6D4, 0x92C9E5BC,
		0x48842DEF, 0x80A82E87, 0xD8DD2B3F, 0x10F12857,
		0x9941CD4A, 0x516DCE22, 0x0918CB9A, 0xC134C8F2,
		0x7A07B77A, 0xB22BB412, 0xEA5EB1AA, 0x2272B2C2,
		0xABC257DF, 0x63EE54B7, 0x3B9B510F, 0xF3B75267,
		0x29FA9A34, 0xE1D6995C, 0xB9A39CE4, 0x718F9F8C,
		0xF83F7A91, 0x301379F9, 0x68667C41, 0xA04A7F29,
		0xDCFDECE6, 0x14D1EF8E, 0x4CA4EA36, 0x8488E95E,
		0x0D380C43, 0xC5140F2B, 0x9D610A93, 0x554D09FB,
		0x8F00C1A8, 0x472CC2C0, 0x1F59C778, 0xD775C410,
		0x5EC5210D, 0x96E92265, 0xCE9C27DD, 0x06B024B5,
		0xC784EC47, 0x0FA8EF2F, 0x57DDEA97, 0x9FF1E9FF,
		0x16410CE2, 0xDE6D0F8A, 0x86180A32, 0x4E34095A,
		0x9479C109, 0x5C55C261, 0x0420C7D9, 0xCC0CC4B1,
		0x45BC21AC, 0x8D9022C4, 0xD5E5277C, 0x1DC92414,
		0x617EB7DB, 0xA952B4B3, 0xF127B10B, 0x390BB263,
		0xB0BB577E, 0x78975416, 0x20E251AE, 0xE8CE52C6,
		0x32839A95, 0xFAAF99FD, 0xA2DA9C45, 0x6AF69F2D,
		0xE3467A30, 0x2B6A7958, 0x731F7CE0, 0xBB337F88,
		0xF40E6EF5, 0x3C226D9D, 0x64576825, 0xAC7B6B4D,
		0x25CB8E50, 0xEDE78D38, 0xB5928880, 0x7DBE8BE8,
		0xA7F343BB, 0x6FDF40D3, 0x37AA456B, 0xFF864603,
		0x7636A31E, 0xBE1AA076, 0xE66FA5CE, 0x2E43A6A6,
		0x52F43569, 0x9AD83601, 0xC2AD33B9, 0x0A8130D1,
		0x8331D5CC, 0x4B1DD6A4, 0x1368D31C, 0xDB44D074,
		0x01091827, 0xC9251B4F, 0x91501EF7, 0x597C1D9F,
		0xD0CCF882, 0x18E0FBEA, 0x4095FE52, 0x88B9FD3A,
		0x498D35C8, 0x81A136A0, 0xD9D43318, 0x11F83070,
		0x9848D56D, 0x5064D605, 0x0811D3BD, 0xC03DD0D5,
		0x1A701886, 0xD25C1BEE, 0x8A291E56, 0x42051D3E,
		0xCBB5F823, 0x0399FB4B, 0x5BECFEF3, 0x93C0FD9B,
		0xEF776E54, 0x275B6D3C, 0x7F2E6884, 0xB7026BEC,
		0x3EB28EF1, 0xF69E8D99, 0xAEEB8821, 0x66C78B49,
		0xBC8A431A, 0x74A64072, 0x2CD345CA, 0xE4FF46A2,
		0x6D4FA3BF, 0xA563A0D7, 0xFD16A56F, 0x353AA607,
		0x8E09D98F, 0x4625DAE7, 0x1E50DF5F, 0xD67CDC37,
		0x5FCC392A, 0x97E03A42, 0xCF953FFA, 0x07B93C92,
		0xDDF4F4C1, 0x15D8F7A9, 0x4DADF211, 0x8581F179,
		0x0C311464, 0xC41D170C, 0x9C6812B4, 0x544411DC,
		0x28F38213, 0xE0DF817B, 0xB8AA84C3, 0x708687AB,
		0xF93662B6, 0x311A61DE, 0x696F6466, 0xA143670E,
		0x7B0EAF5D, 0xB322AC35, 0xEB57A98D, 0x237BAAE5,
		0xAACB4FF8, 0x62E74C90, 0x3A924928, 0xF2BE4A40,
		0x338A82B2, 0xFBA681DA, 0xA3D38462, 0x6BFF870A,
		0xE24F6217, 0x2A63617F, 0x721664C7, 0xBA3A67AF,
		0x6077AFFC, 0xA85BAC94, 0xF02EA92C, 0x3802AA44,
		0xB1B24F59, 0x799E4C31, 0x21EB4989, 0xE9C74AE1,
		0x9570D92E, 0x5D5CDA46, 0x0529DFFE, 0xCD05DC96,
		0x44B5398B, 0x8C993AE3, 0xD4EC3F5B, 0x1CC03C33,
		0xC68DF460, 0x0EA1F708, 0x56D4F2B0, 0x9EF8F1D8,
		0x174814C5, 0xDF6417AD, 0x87111215, 0x4F3D117D,
	},
	{
		0x00000000, 0x277D3C49, 0x4EFA7892, 0x698744DB,
		0x6D821D21, 0x4AFF2168, 0x237865B3, 0x040559FA,
		0xDA043B42, 0xFD79070B, 0x94FE43D0, 0xB3837F99,
		0xB7862663, 0x90FB1A2A, 0xF97C5EF1, 0xDE0162B8,
		0xB4097684, 0x93744ACD, 0xFAF30E16, 0xDD8E325F,
		0xD98B6BA5, 0xFEF657EC, 0x97711337, 0xB00C2F7E,
		0x6E0D4DC6, 0x4970718F, 0x20F73554, 0x078A091D,
		0x038F50E7, 0x24F26CAE, 0x4D752875, 0x6A08143C,
		0x9965000D, 0xBE183C44, 0xD79F789F, 0xF0E244D6,
		0xF4E71D2C, 0xD39A2165, 0xBA1D65BE, 0x9D6059F7,
		0x43613B4F, 0x641C0706, 0x0D9B43DD, 0x2AE67F94,
		0x2EE3266E, 0x099E1A27, 0x60195EFC, 0x476462B5,
		0x2D6C7689, 0x0A114AC0, 0x63960E1B, 0x44EB3252,
		0x40EE6BA8, 0x679357E1, 0x0E14133A, 0x29692F73,
		0xF7684DCB, 0xD0157182, 0xB9923559, 0x9EEF0910,
		0x9AEA50EA, 0xBD976CA3, 0xD4102878, 0xF36D1431,
		0x32CB001A, 0x15B63C53, 0x7C317888, 0x5B4C44C1,
		0x5F491D3B, 0x78342172, 0x11B365A9, 0x36CE59E0,
		0xE8CF3B58, 0xCFB20711, 0xA63543CA, 0x81487F83,
		0x854D2679, 0xA2301A30, 0xCBB75EEB, 0xECCA62A2,
		0x86C2769E, 0xA1BF4AD7, 0xC8380E0C, 0xEF453245,
		0xEB406BBF, 0xCC3D57F6, 0xA5BA132D, 0x82C72F64,
		0x5CC64DDC, 0x7BBB7195, 0x123C354E, 0x35410907,
		0x314450FD, 0x16396CB4, 0x7FBE286F, 0x58C31426,
		0xABAE0017, 0x8CD33C5E, 0xE5547885, 0xC22944CC,
		0xC62C1D36, 0xE151217F, 0x88D665A4, 0xAFAB59ED,
		0x71AA3B55, 0x56D7071C, 0x3F5043C7, 0x182D7F8E,
		0x1C282674, 0x3B551A3D, 0x52D25EE6, 0x75AF62AF,
		0x1FA77693, 0x38DA4ADA, 0x515D0E01, 0x76203248,
		0x72256BB2, 0x555857FB, 0x3CDF1320, 0x1BA22F69,
		0xC5A34DD1, 0xE2DE7198, 0x8B593543, 0xAC24090A,
		0xA82150F0, 0x8F5C6CB9, 0xE6DB2862, 0xC1A6142B,
		0x64960134, 0x43EB3D7D, 0x2A6C79A6, 0x0D1145EF,
		0x09141C15, 0x2E69205C, 0x47EE6487, 0x609358CE,
		0xBE923A76, 0x99EF063F, 0xF06842E4, 0xD7157EAD,
		0xD3102757, 0xF46D1B1E, 0x9DEA5FC5, 0xBA97638C,
		0xD09F77B0, 0xF7E24BF9, 0x9E650F22, 0xB918336B,
		0xBD1D6A91, 0x9A6056D8, 0xF3E71203, 0xD49A2E4A,
		0x0A9B4CF2, 0x2DE670BB, 0x44613460, 0x631C0829,
		0x671951D3, 0x40646D9A, 0x29E32941, 0x0E9E1508,
		0xFDF30139, 0xDA8E3D70, 0xB30979AB, 0x947445E2,
		0x90711C18, 0xB70C2051, 0xDE8B648A, 0xF9F658C3,
		0x27F73A7B, 0x008A0632, 0x690D42E9, 0x4E707EA0,
		0x4A75275A, 0x6D081B13, 0x048F5FC8, 0x23F26381,
		0x49FA77BD, 0x6E874BF4, 0x07000F2F, 0x207D3366,
		0x24786A9C, 0x030556D5, 0x6A82120E, 0x4DFF2E47,
		0x93FE4CFF, 0xB48370B6, 0xDD04346D, 0xFA790824,
		0xFE7C51DE, 0xD9016D97, 0xB086294C, 0x97FB1505,
		0x565D012E, 0x71203D67, 0x18A779BC, 0x3FDA45F5,
		0x3BDF1C0F, 0x1CA22046, 0x7525649D, 0x525858D4,
		0x8C593A6C, 0xAB240625, 0xC2A342FE, 0xE5DE7EB7,
		0xE1DB274D, 0xC6A61B04, 0xAF215FDF, 0x885C6396,
		0xE25477AA, 0xC5294BE3, 0xACAE0F38, 0x8BD33371,
		0x8FD66A8B, 0xA8AB56C2, 0xC12C1219, 0xE6512E50,
		0x38504CE8, 0x1F2D70A1, 0x76AA347A, 0x51D70833,
		0x55D251C9, 0x72AF6D80, 0x1B28295B, 0x3C551512,
		0xCF380123, 0xE8453D6A, 0x81C279B1, 0xA6BF45F8,
		0xA2BA1C02, 0x85C7204B, 0xEC406490, 0xCB3D58D9,
		0x153C3A61, 0x32410628, 0x5BC642F3, 0x7CBB7EBA,
		0x78BE2740, 0x5FC31B09, 0x36445FD2, 0x1139639B,
		0x7B3177A7, 0x5C4C4BEE, 0x35CB0F35, 0x12B6337C,
		0x16B36A86, 0x31CE56CF, 0x58491214, 0x7F342E5D,
		0xA1354CE5, 0x864870AC, 0xEFCF3477, 0xC8B2083E,
		0xCCB751C4, 0xEBCA6D8D, 0x824D2956, 0xA530151F
	}
#endif							/* WORDS_BIGENDIAN */
};
